<?php

namespace Database\Seeders;

use App\Models\Siswa;
use App\Models\BookHead;
use App\Models\BookDetail;
use App\Models\MapelHead;
use App\Models\MapelDetail;
use App\Models\MapelKelas;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class BookHeadDetailSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        Log::info('Starting BookHeadDetailSeeder...');
        
        DB::beginTransaction();
        
        try {
            // Get all active students
            $students = Siswa::where('deleted', '0')
                ->where('status', 'Aktif')
                ->with('kelas')
                ->get();

            Log::info("Found {$students->count()} active students");

            if ($students->isEmpty()) {
                Log::warning('No active students found. Seeder stopped.');
                $this->command->warn('No active students found. Please seed students first.');
                return;
            }

            $totalBookHeadsCreated = 0;
            $totalBookDetailsCreated = 0;

            foreach ($students as $siswa) {
                Log::info("Processing student: {$siswa->nama_siswa} (NISN: {$siswa->nisn})");

                // Get tahun ajaran
                $thnAjaranAwal = $siswa->thn_ajaran_awal;
                $thnAjaranAkhir = $siswa->thn_ajaran_akhir;

                if (!$thnAjaranAwal || !$thnAjaranAkhir) {
                    Log::warning("Student {$siswa->nama_siswa} has no tahun ajaran. Skipping.");
                    continue;
                }

                // Get mapel that are assigned to this student's class
                $mapelAssignments = MapelKelas::where('kelas_id', $siswa->kelas_id)
                    ->get();

                if ($mapelAssignments->isEmpty()) {
                    Log::warning("No mapel assigned to class {$siswa->kelas->nama_kelas_nomor}. Skipping student {$siswa->nama_siswa}.");
                    continue;
                }

                // Separate mapel heads and details
                $mapelHeadIds = $mapelAssignments->where('mapel_type', 'head')->pluck('mapel_id')->toArray();
                $mapelDetailIds = $mapelAssignments->where('mapel_type', 'detail')->pluck('mapel_id')->toArray();

                // Get MapelHead with their details
                $mapelHeads = MapelHead::with(['details' => function ($query) use ($mapelDetailIds) {
                    $query->where('status', 'Aktif')
                        ->where('deleted', '0')
                        ->whereIn('id', $mapelDetailIds);
                }])
                    ->where('status', 'Aktif')
                    ->where('deleted', '0')
                    ->whereIn('id', $mapelHeadIds)
                    ->get();

                if ($mapelHeads->isEmpty()) {
                    Log::warning("No active mapel found for student {$siswa->nama_siswa}. Skipping.");
                    continue;
                }

                Log::info("Found {$mapelHeads->count()} mapel heads for student {$siswa->nama_siswa}");

                // Create BookHead for both semesters
                $semesters = ['ganjil', 'genap'];
                
                foreach ($semesters as $semester) {
                    // Check if BookHead already exists
                    $existingBookHead = BookHead::where('siswa_id', $siswa->id)
                        ->where('thn_ajaran_awal', $thnAjaranAwal)
                        ->where('thn_ajaran_akhir', $thnAjaranAkhir)
                        ->where('semester', ucfirst($semester))
                        ->where('deleted', '0')
                        ->first();

                    if ($existingBookHead) {
                        Log::info("BookHead already exists for {$siswa->nama_siswa} - Semester {$semester}. Skipping.");
                        continue;
                    }

                    // Create BookHead
                    $bookHead = BookHead::create([
                        'siswa_id' => $siswa->id,
                        'thn_ajaran_awal' => $thnAjaranAwal,
                        'thn_ajaran_akhir' => $thnAjaranAkhir,
                        'nama_kelas_nomor' => $siswa->kelas->nama_kelas_nomor ?? null,
                        'semester' => ucfirst($semester),
                        'jml_nilai_akhir' => 0,
                        'deleted' => '0',
                    ]);

                    $totalBookHeadsCreated++;
                    Log::info("Created BookHead for {$siswa->nama_siswa} - Semester {$semester}");

                    $totalNilai = 0;
                    $bookDetailsCount = 0;

                    // Create BookDetail for each mapel
                    foreach ($mapelHeads as $mapelHead) {
                        if ($mapelHead->details->isNotEmpty()) {
                            // Mapel has details, create BookDetail for each detail
                            foreach ($mapelHead->details as $detail) {
                                $nilai = $this->generateNilai();
                                $deskripsi = $this->generateDeskripsi($detail->nama_mapel_detail, $nilai);

                                BookDetail::create([
                                    'book_head_id' => $bookHead->id,
                                    'mapel_head_id' => null,
                                    'mapel_detail_id' => $detail->id,
                                    'nilai_akhir' => $nilai,
                                    'deskripsi' => $deskripsi,
                                    'deleted' => '0',
                                ]);

                                $totalNilai += $nilai;
                                $bookDetailsCount++;
                                $totalBookDetailsCreated++;
                            }
                        } else {
                            // Mapel has no details, create BookDetail for the head
                            $nilai = $this->generateNilai();
                            $deskripsi = $this->generateDeskripsi($mapelHead->nama_mapel, $nilai);

                            BookDetail::create([
                                'book_head_id' => $bookHead->id,
                                'mapel_head_id' => $mapelHead->id,
                                'mapel_detail_id' => null,
                                'nilai_akhir' => $nilai,
                                'deskripsi' => $deskripsi,
                                'deleted' => '0',
                            ]);

                            $totalNilai += $nilai;
                            $bookDetailsCount++;
                            $totalBookDetailsCreated++;
                        }
                    }

                    // Update jml_nilai_akhir
                    $bookHead->update([
                        'jml_nilai_akhir' => $totalNilai
                    ]);

                    Log::info("Created {$bookDetailsCount} BookDetails for {$siswa->nama_siswa} - Semester {$semester} with total nilai: {$totalNilai}");
                }
            }

            DB::commit();
            
            $this->command->info("✅ Seeder completed successfully!");
            $this->command->info("📊 Summary:");
            $this->command->info("   - Total BookHeads created: {$totalBookHeadsCreated}");
            $this->command->info("   - Total BookDetails created: {$totalBookDetailsCreated}");
            
            Log::info("BookHeadDetailSeeder completed: {$totalBookHeadsCreated} BookHeads, {$totalBookDetailsCreated} BookDetails created");
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error in BookHeadDetailSeeder: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());
            $this->command->error('❌ Seeder failed: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Generate random nilai between 70-100
     */
    private function generateNilai(): int
    {
        return rand(70, 100);
    }

    /**
     * Generate deskripsi based on mapel name and nilai
     */
    private function generateDeskripsi(string $mapelName, int $nilai): string
    {
        $deskripsiTemplates = [];

        if ($nilai >= 90) {
            $deskripsiTemplates = [
                "Siswa menunjukkan pemahaman yang sangat baik dalam {$mapelName}. Terus pertahankan prestasi yang luar biasa ini.",
                "Prestasi yang sangat memuaskan dalam {$mapelName}. Siswa menguasai materi dengan sangat baik.",
                "Siswa memiliki kemampuan yang sangat baik dalam {$mapelName}. Teruskan kerja keras dan dedikasimu.",
                "Penguasaan materi {$mapelName} sangat baik. Siswa aktif dan menunjukkan pemahaman yang mendalam.",
            ];
        } elseif ($nilai >= 80) {
            $deskripsiTemplates = [
                "Siswa menunjukkan pemahaman yang baik dalam {$mapelName}. Pertahankan dan tingkatkan terus prestasimu.",
                "Prestasi yang baik dalam {$mapelName}. Siswa memahami sebagian besar materi dengan baik.",
                "Siswa memiliki kemampuan yang baik dalam {$mapelName}. Tingkatkan lagi agar menjadi lebih baik.",
                "Penguasaan materi {$mapelName} cukup baik. Terus belajar dengan giat.",
            ];
        } else {
            $deskripsiTemplates = [
                "Siswa menunjukkan pemahaman yang cukup dalam {$mapelName}. Perlu lebih giat belajar lagi.",
                "Prestasi yang cukup dalam {$mapelName}. Siswa perlu meningkatkan pemahaman materi.",
                "Siswa memiliki kemampuan yang cukup dalam {$mapelName}. Tingkatkan lagi dengan belajar lebih giat.",
                "Penguasaan materi {$mapelName} cukup. Perlu bimbingan tambahan untuk meningkatkan pemahaman.",
            ];
        }

        return $deskripsiTemplates[array_rand($deskripsiTemplates)];
    }
}
