<?php

namespace App\Imports;

use App\Models\Siswa;
use App\Models\Kelas;
use App\Models\Pekerjaan;
use Illuminate\Support\Collection;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\WithCalculatedFormulas;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class SiswaImport implements ToCollection, WithCalculatedFormulas
{
    protected $errors = [];
    protected $successCount = 0;
    protected $updateCount = 0;
    protected $headerRow = null;

    public function collection(Collection $rows)
    {
        DB::beginTransaction();
        
        try {
            if ($rows->isEmpty()) {
                Log::warning("No rows to process");
                return;
            }
            
            // First row is header
            $this->headerRow = $rows->first();
            
            Log::info("Total rows to process: " . ($rows->count() - 1));
            
            // Process data rows (skip header)
            foreach ($rows->skip(1) as $index => $row) {
                $rowNumber = $index + 2;
                
                // Get data from Excel columns (sesuai format export)
                // 0: No (skip)
                $kelas_name = isset($row[1]) ? trim((string) $row[1]) : null;
                $tahun_ajaran = isset($row[2]) ? trim((string) $row[2]) : null;
                $nis = isset($row[3]) ? trim((string) $row[3]) : null;
                $nisn = isset($row[4]) ? trim((string) $row[4]) : null;
                $no_kip = isset($row[5]) ? trim((string) $row[5]) : null;
                $no_kk = isset($row[6]) ? trim((string) $row[6]) : null;
                $nama_siswa = isset($row[7]) ? trim((string) $row[7]) : null;
                $jenis_kelamin = isset($row[8]) ? trim((string) $row[8]) : null;
                $tempat_lahir = isset($row[9]) ? trim((string) $row[9]) : null;
                $tanggal_lahir = isset($row[10]) ? $this->parseDate($row[10]) : null;
                $agama = isset($row[11]) ? trim((string) $row[11]) : null;
                $alamat = isset($row[12]) ? trim((string) $row[12]) : null;
                $no_hp_siswa = isset($row[13]) ? trim((string) $row[13]) : null;
                $nama_ayah = isset($row[14]) ? trim((string) $row[14]) : null;
                $pekerjaan_ayah = isset($row[15]) ? trim((string) $row[15]) : null;
                $nama_ibu = isset($row[16]) ? trim((string) $row[16]) : null;
                $pekerjaan_ibu = isset($row[17]) ? trim((string) $row[17]) : null;
                $no_hp_ortu = isset($row[18]) ? trim((string) $row[18]) : null;
                $angkatan = isset($row[19]) ? trim((string) $row[19]) : null;
                
                // Default values untuk field yang tidak ada di export
                $status = 'Aktif';
                $deleted = '0';
                $foto = null;
                $keterangan_kenaikan = null;
                
                Log::info("Processing row {$rowNumber}, NISN: {$nisn}, Nama: {$nama_siswa}");
                
                // Skip empty rows (check NISN and nama_siswa as minimum required fields)
                if (empty($nisn) || empty($nama_siswa)) {
                    Log::info("Skipping row {$rowNumber} - empty NISN or nama_siswa");
                    continue;
                }

                // Validate required fields
                if (empty($nis)) {
                    $this->errors[] = "Baris {$rowNumber}: NIS tidak boleh kosong";
                    continue;
                }

                if (empty($jenis_kelamin) || !in_array($jenis_kelamin, ['Laki-Laki', 'Perempuan'])) {
                    $this->errors[] = "Baris {$rowNumber}: Jenis kelamin harus 'Laki-Laki' atau 'Perempuan'";
                    continue;
                }

                if (empty($agama) || !in_array($agama, ['Islam', 'Kristen', 'Hindu', 'Buddha', 'Konghucu'])) {
                    $this->errors[] = "Baris {$rowNumber}: Agama tidak valid (harus: Islam, Kristen, Hindu, Buddha, atau Konghucu)";
                    continue;
                }

                if (empty($kelas_name)) {
                    $this->errors[] = "Baris {$rowNumber}: Kelas tidak boleh kosong";
                    continue;
                }

                if (empty($angkatan)) {
                    $this->errors[] = "Baris {$rowNumber}: Angkatan tidak boleh kosong";
                    continue;
                }

                if (empty($tahun_ajaran)) {
                    $this->errors[] = "Baris {$rowNumber}: Tahun ajaran tidak boleh kosong";
                    continue;
                }

                // Find Kelas
                $kelas = $this->findKelas($kelas_name);
                if (!$kelas) {
                    $this->errors[] = "Baris {$rowNumber}: Kelas '{$kelas_name}' tidak ditemukan";
                    continue;
                }

                // Find Pekerjaan IDs
                $pekerjaan_ayah_id = $pekerjaan_ayah ? $this->findPekerjaanId($pekerjaan_ayah) : null;
                $pekerjaan_ibu_id = $pekerjaan_ibu ? $this->findPekerjaanId($pekerjaan_ibu) : null;

                // Parse tahun ajaran
                $tahunAjaranParts = explode('/', $tahun_ajaran);
                if (count($tahunAjaranParts) !== 2) {
                    $this->errors[] = "Baris {$rowNumber}: Format tahun ajaran tidak valid (harus: 2025/2026)";
                    continue;
                }
                $thn_ajaran_awal = (int) $tahunAjaranParts[0];
                $thn_ajaran_akhir = (int) $tahunAjaranParts[1];

                // Check if siswa with this NISN already exists
                $existingSiswa = Siswa::where('nisn', $nisn)->where('deleted', '0')->first();

                if ($existingSiswa) {
                    // Update existing siswa
                    $existingSiswa->update([
                        'nis' => $nis,
                        'no_kip' => $no_kip,
                        'no_kk' => $no_kk,
                        'nama_siswa' => $nama_siswa,
                        'jenis_kelamin' => $jenis_kelamin,
                        'tempat_lahir' => $tempat_lahir,
                        'tanggal_lahir' => $tanggal_lahir,
                        'agama' => $agama,
                        'alamat' => $alamat,
                        'no_hp_siswa' => $no_hp_siswa,
                        'nama_ayah' => $nama_ayah,
                        'pekerjaan_ayah_id' => $pekerjaan_ayah_id,
                        'nama_ibu' => $nama_ibu,
                        'pekerjaan_ibu_id' => $pekerjaan_ibu_id,
                        'no_hp_ortu' => $no_hp_ortu,
                        'kelas_id' => $kelas->id,
                        'angkatan' => $angkatan,
                        'status' => $status,
                        'tahun_ajaran' => $tahun_ajaran,
                        'thn_ajaran_awal' => $thn_ajaran_awal,
                        'thn_ajaran_akhir' => $thn_ajaran_akhir,
                        'keterangan_kenaikan' => $keterangan_kenaikan,
                    ]);
                    
                    Log::info("Updated siswa: {$nama_siswa} (NISN: {$nisn})");
                    $this->updateCount++;
                } else {
                    // Check for duplicate NIS
                    $duplicateNIS = Siswa::where('nis', $nis)->where('deleted', '0')->exists();
                    if ($duplicateNIS) {
                        $this->errors[] = "Baris {$rowNumber}: NIS '{$nis}' sudah terdaftar";
                        continue;
                    }

                    // Check for duplicate No KIP if provided
                    if ($no_kip) {
                        $duplicateKIP = Siswa::where('no_kip', $no_kip)->where('deleted', '0')->exists();
                        if ($duplicateKIP) {
                            $this->errors[] = "Baris {$rowNumber}: No KIP '{$no_kip}' sudah terdaftar";
                            continue;
                        }
                    }

                    // Create new siswa
                    Siswa::create([
                        'nis' => $nis,
                        'nisn' => $nisn,
                        'no_kip' => $no_kip,
                        'no_kk' => $no_kk,
                        'nama_siswa' => $nama_siswa,
                        'jenis_kelamin' => $jenis_kelamin,
                        'tempat_lahir' => $tempat_lahir,
                        'tanggal_lahir' => $tanggal_lahir,
                        'agama' => $agama,
                        'alamat' => $alamat,
                        'no_hp_siswa' => $no_hp_siswa,
                        'nama_ayah' => $nama_ayah,
                        'pekerjaan_ayah_id' => $pekerjaan_ayah_id,
                        'nama_ibu' => $nama_ibu,
                        'pekerjaan_ibu_id' => $pekerjaan_ibu_id,
                        'no_hp_ortu' => $no_hp_ortu,
                        'kelas_id' => $kelas->id,
                        'angkatan' => $angkatan,
                        'status' => $status,
                        'tahun_ajaran' => $tahun_ajaran,
                        'thn_ajaran_awal' => $thn_ajaran_awal,
                        'thn_ajaran_akhir' => $thn_ajaran_akhir,
                        'deleted' => $deleted,
                        'foto' => $foto,
                        'keterangan_kenaikan' => $keterangan_kenaikan,
                    ]);
                    
                    Log::info("Created new siswa: {$nama_siswa} (NISN: {$nisn})");
                    $this->successCount++;
                }
            }

            DB::commit();
            
            Log::info("Import completed - Created: {$this->successCount}, Updated: {$this->updateCount}, Errors: " . count($this->errors));
        } catch (\Exception $e) {
            DB::rollBack();
            $this->errors[] = "Error: " . $e->getMessage();
            Log::error("Import error: " . $e->getMessage());
        }
    }

    protected function findKelas($kelasName)
    {
        // Try to parse kelas name (e.g., "7A", "8", "9B")
        $kelasName = strtoupper(trim($kelasName));
        
        // Extract nomor and sub
        preg_match('/^(\d+)([A-Z]?)$/', $kelasName, $matches);
        
        if (empty($matches)) {
            return null;
        }
        
        $nomor = $matches[1];
        $sub = isset($matches[2]) && $matches[2] !== '' ? $matches[2] : null;
        
        // Find kelas
        $query = Kelas::where('nama_kelas_nomor', $nomor)
            ->where('deleted', '0');
        
        if ($sub) {
            $query->where('nama_kelas_sub', $sub);
        } else {
            $query->whereNull('nama_kelas_sub');
        }
        
        return $query->first();
    }

    protected function findPekerjaanId($pekerjaanName)
    {
        $pekerjaan = Pekerjaan::where('nama_pekerjaan', 'like', '%' . $pekerjaanName . '%')
            ->where('deleted', '0')
            ->first();
        
        return $pekerjaan ? $pekerjaan->id : null;
    }

    protected function parseDate($date)
    {
        if (empty($date)) {
            return null;
        }

        // If it's a numeric Excel date
        if (is_numeric($date)) {
            try {
                $unixDate = ($date - 25569) * 86400;
                return date('Y-m-d', $unixDate);
            } catch (\Exception $e) {
                return null;
            }
        }

        // Try to parse common date formats
        try {
            $parsedDate = \Carbon\Carbon::parse($date);
            return $parsedDate->format('Y-m-d');
        } catch (\Exception $e) {
            return null;
        }
    }

    public function getErrors()
    {
        return $this->errors;
    }

    public function getSuccessCount()
    {
        return $this->successCount;
    }

    public function getUpdateCount()
    {
        return $this->updateCount;
    }
}
