<?php

namespace App\Imports;

use App\Models\BookHead;
use App\Models\BookDetail;
use App\Models\Siswa;
use App\Models\MapelHead;
use App\Models\MapelDetail;
use Illuminate\Support\Collection;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\WithStartRow;
use Maatwebsite\Excel\Concerns\WithCalculatedFormulas;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Maatwebsite\Excel\Concerns\WithHeadingRow;

class BukuIndukImport implements ToCollection, WithCalculatedFormulas
{
    protected $semester;
    protected $kelasId;
    protected $thnAjaranAwal;
    protected $thnAjaranAkhir;
    protected $mapelHeads;
    protected $errors = [];
    protected $successCount = 0;
    protected $headerRow = null;
    protected $headerRow2 = null;
    protected $headerRow3 = null;
    protected $mapelMapping = [];

    public function __construct($semester, $kelasId, $thnAjaranAwal, $thnAjaranAkhir)
    {
        $this->semester = $semester;
        $this->kelasId = $kelasId;
        $this->thnAjaranAwal = $thnAjaranAwal;
        $this->thnAjaranAkhir = $thnAjaranAkhir;
        
        // Load mapel heads and details for this kelas
        $this->loadMapelData();
    }

    protected function loadMapelData()
    {
        // Get mapel heads that are assigned to this kelas
        $mapelHeadIds = \App\Models\MapelKelas::where('kelas_id', $this->kelasId)
            ->where('mapel_type', 'head')
            ->pluck('mapel_id');

        // Get mapel detail ids that are assigned to this kelas
        $mapelDetailIds = \App\Models\MapelKelas::where('kelas_id', $this->kelasId)
            ->where('mapel_type', 'detail')
            ->pluck('mapel_id');

        $this->mapelHeads = MapelHead::with(['details' => function ($query) use ($mapelDetailIds) {
            $query->where('status', 'Aktif')
                ->where('deleted', '0')
                ->whereIn('id', $mapelDetailIds)
                ->orderBy('nama_mapel_detail');
        }])
            ->where('status', 'Aktif')
            ->where('deleted', '0')
            ->whereIn('id', $mapelHeadIds)
            ->orderBy('nama_mapel')
            ->get();
    }

    protected function buildMapelMapping($headerRow1, $headerRow2, $headerRow3)
    {
        // Build mapping between column index and mapel/detail
        // Header format dari export:
        // Row 1: No | NISN | Nama Siswa | MapelHead/Detail | (empty for colspan) | ...
        // Row 2: (empty) | (empty) | (empty) | DetailName | (empty for colspan) | ... (if has details)
        // Row 3: (empty) | (empty) | (empty) | Nilai | Deskripsi | Nilai | Deskripsi | ...
        
        $mapping = [];
        $headerArray1 = $headerRow1->toArray();
        $headerArray2 = $headerRow2->toArray();
        $headerArray3 = $headerRow3->toArray();
        
        Log::info("Building mapel mapping from 3 header rows");
        Log::info("Header Row 1: " . json_encode($headerArray1));
        Log::info("Header Row 2: " . json_encode($headerArray2));
        Log::info("Header Row 3: " . json_encode($headerArray3));
        
        // Find all mapel/detail columns and their nilai/deskripsi columns
        // Start from column 3 (after No, NISN, Nama Siswa)
        for ($colIndex = 3; $colIndex < count($headerArray3); $colIndex++) {
            // Check row 3 for "Nilai" label to identify data columns
            $row3Value = trim($headerArray3[$colIndex] ?? '');
            $row3ValueLower = strtolower($row3Value);
            
            // Only process columns that have "Nilai" in row 3
            if ($row3ValueLower !== 'nilai') {
                continue;
            }
            
            // Nilai column found, next column should be Deskripsi
            $nilaiCol = $colIndex;
            $deskripsiCol = $colIndex + 1;
            
            // Verify next column is "Deskripsi"
            $nextValue = trim($headerArray3[$deskripsiCol] ?? '');
            if (strtolower($nextValue) !== 'deskripsi') {
                Log::warning("Expected 'Deskripsi' at column {$deskripsiCol}, found: {$nextValue}");
                continue;
            }
            
            // Get mapel/detail name from row 2 (detail name) or row 1 (mapel head name)
            $row2Value = trim($headerArray2[$colIndex] ?? '');
            $row1Value = trim($headerArray1[$colIndex] ?? '');
            
            Log::info("Column {$colIndex} - Row1: '{$row1Value}', Row2: '{$row2Value}', Row3: '{$row3Value}'");
            
            // Try to match with mapel/detail
            $matched = false;
            
            foreach ($this->mapelHeads as $mapelHead) {
                $mapelHeadName = trim($mapelHead->nama_mapel);
                $mapelHeadNameLower = strtolower($mapelHeadName);
                
                if ($mapelHead->details->isNotEmpty()) {
                    // Has details - check row 2 for detail name
                    foreach ($mapelHead->details as $detail) {
                        $detailName = trim($detail->nama_mapel_detail);
                        $detailNameLower = strtolower($detailName);
                        
                        // Match by detail name in row 2
                        if (!empty($row2Value) && (
                            $row2Value === $detailName || 
                            strtolower($row2Value) === $detailNameLower ||
                            str_contains(strtolower($row2Value), $detailNameLower) ||
                            str_contains($detailNameLower, strtolower($row2Value)))) {
                            
                            $mapping[$colIndex] = [
                                'type' => 'detail',
                                'mapel_head_id' => $mapelHead->id,
                                'mapel_detail_id' => $detail->id,
                                'mapel_name' => $mapelHeadName,
                                'detail_name' => $detailName,
                                'nilai_col' => $nilaiCol,
                                'deskripsi_col' => $deskripsiCol
                            ];
                            
                            Log::info("Mapped detail: {$detailName} (from {$mapelHeadName}) - Nilai col: {$nilaiCol}, Deskripsi col: {$deskripsiCol}");
                            $matched = true;
                            break 2;
                        }
                    }
                } else {
                    // No details - check row 1 for mapel head name
                    if (!empty($row1Value) && (
                        $row1Value === $mapelHeadName || 
                        strtolower($row1Value) === $mapelHeadNameLower ||
                        str_contains(strtolower($row1Value), $mapelHeadNameLower) ||
                        str_contains($mapelHeadNameLower, strtolower($row1Value)))) {
                        
                        $mapping[$colIndex] = [
                            'type' => 'head',
                            'mapel_head_id' => $mapelHead->id,
                            'mapel_detail_id' => null,
                            'mapel_name' => $mapelHeadName,
                            'detail_name' => null,
                            'nilai_col' => $nilaiCol,
                            'deskripsi_col' => $deskripsiCol
                        ];
                        
                        Log::info("Mapped head: {$mapelHeadName} - Nilai col: {$nilaiCol}, Deskripsi col: {$deskripsiCol}");
                        $matched = true;
                        break;
                    }
                }
            }
            
            if (!$matched) {
                Log::warning("Could not match column {$colIndex} (Row1: '{$row1Value}', Row2: '{$row2Value}')");
            }
        }
        
        $this->mapelMapping = $mapping;
        Log::info("Total mappings created: " . count($mapping));
        
        return $mapping;
    }

    public function collection(Collection $rows)
    {
        DB::beginTransaction();
        
        try {
            if ($rows->isEmpty()) {
                Log::warning("No rows to process");
                return;
            }
            
            // First 3 rows are headers
            if ($rows->count() < 4) {
                Log::warning("Not enough rows (need at least 3 header rows + 1 data row)");
                return;
            }
            
            $this->headerRow = $rows->get(0);
            $this->headerRow2 = $rows->get(1);
            $this->headerRow3 = $rows->get(2);
            $this->buildMapelMapping($this->headerRow, $this->headerRow2, $this->headerRow3);
            
            Log::info("Total rows to process: " . ($rows->count() - 3)); // -3 for 3 header rows
            
            // Process data rows (skip 3 header rows)
            foreach ($rows->skip(3) as $index => $row) {
                $rowNumber = $index + 4; // +4 karena mulai dari row 4 (after 3 header rows)
                
                // Get NISN from column B (index 1)
                $nisn = isset($row[1]) ? trim((string) $row[1]) : null;
                
                Log::info("Processing row {$rowNumber}, NISN: {$nisn}");
                
                // Skip empty rows
                if (empty($nisn) || $nisn === '') {
                    Log::info("Skipping row {$rowNumber} - empty NISN");
                    continue;
                }

                // Find siswa by NISN - use the first/oldest record for consistency across years
                $siswa = Siswa::where('nisn', $nisn)
                    ->where('deleted', '0')
                    ->orderBy('id', 'asc')
                    ->first();

                if (!$siswa) {
                    $this->errors[] = "Baris {$rowNumber}: NISN {$nisn} tidak ditemukan";
                    Log::warning("NISN {$nisn} not found in database");
                    continue;
                }
                
                // Validate that siswa exists in the specified class and academic year
                $siswaInKelas = Siswa::where('nisn', $nisn)
                    ->where('kelas_id', $this->kelasId)
                    ->where('thn_ajaran_awal', $this->thnAjaranAwal)
                    ->where('thn_ajaran_akhir', $this->thnAjaranAkhir)
                    ->where('deleted', '0')
                    ->first();
                    
                if (!$siswaInKelas) {
                    $this->errors[] = "Baris {$rowNumber}: NISN {$nisn} tidak terdaftar di kelas dan tahun ajaran ini";
                    Log::warning("NISN {$nisn} not found in specified class and academic year");
                    continue;
                }
                
                Log::info("Found siswa: {$siswa->nama_siswa} (ID: {$siswa->id})");

                // Check if BookHead already exists
                $bookHead = BookHead::where('siswa_id', $siswa->id)
                    ->where('thn_ajaran_awal', $this->thnAjaranAwal)
                    ->where('thn_ajaran_akhir', $this->thnAjaranAkhir)
                    ->where('semester', $this->semester)
                    ->where('deleted', '0')
                    ->first();

                if (!$bookHead) {
                    // Create new BookHead
                    $bookHead = BookHead::create([
                        'siswa_id' => $siswa->id,
                        'thn_ajaran_awal' => $this->thnAjaranAwal,
                        'thn_ajaran_akhir' => $this->thnAjaranAkhir,
                        'nama_kelas_nomor' => $siswa->kelas->nama_kelas_nomor ?? null,
                        'semester' => $this->semester,
                        'jml_nilai_akhir' => 0,
                        'deleted' => '0',
                    ]);
                }

                // Process mapel values using mapping
                $this->processMapelValues($bookHead, $row, $rowNumber);

                // Update jml_nilai_akhir with sum of all nilai_akhir from book_details
                $totalNilai = BookDetail::where('book_head_id', $bookHead->id)
                    ->where('deleted', '0')
                    ->sum('nilai_akhir');
                
                $bookHead->update([
                    'jml_nilai_akhir' => $totalNilai ?? 0
                ]);

                Log::info("Updated jml_nilai_akhir for siswa {$siswa->nama_siswa}: {$totalNilai}");

                $this->successCount++;
            }

            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            $this->errors[] = "Error: " . $e->getMessage();
        }
    }

    protected function processMapelValues($bookHead, $row, $rowNumber)
    {
        Log::info("Processing mapel values for row {$rowNumber} using mapping");

        if (empty($this->mapelMapping)) {
            Log::warning("No mapel mapping found, skipping row {$rowNumber}");
            return;
        }

        // Process each mapped mapel/detail
        foreach ($this->mapelMapping as $colIndex => $mapping) {
            $nilaiCol = $mapping['nilai_col'];
            $deskripsiCol = $mapping['deskripsi_col'];
            
            // Get nilai and deskripsi from mapped columns
            $nilaiRaw = $row[$nilaiCol] ?? null;
            $nilai = ($nilaiRaw !== null && $nilaiRaw !== '') ? $nilaiRaw : null;
            
            $deskripsiRaw = $row[$deskripsiCol] ?? null;
            $deskripsi = ($deskripsiRaw !== null && $deskripsiRaw !== '') ? trim((string) $deskripsiRaw) : null;

            $mapelName = $mapping['type'] === 'detail' ? $mapping['detail_name'] : $mapping['mapel_name'];
            Log::info("Processing {$mapping['type']}: {$mapelName} - Nilai col {$nilaiCol}={$nilai}, Deskripsi col {$deskripsiCol}={$deskripsi}");

            // Only save if there's nilai or deskripsi
            if (!is_null($nilai) || !is_null($deskripsi)) {
                if ($mapping['type'] === 'detail') {
                    // Process detail
                    $existingDetail = BookDetail::where('book_head_id', $bookHead->id)
                        ->where('mapel_detail_id', $mapping['mapel_detail_id'])
                        ->first();

                    if ($existingDetail) {
                        $existingDetail->update([
                            'nilai_akhir' => $nilai,
                            'deskripsi' => $deskripsi,
                        ]);
                        Log::info("Updated detail: {$mapelName}");
                    } else {
                        BookDetail::create([
                            'book_head_id' => $bookHead->id,
                            'mapel_head_id' => null,
                            'mapel_detail_id' => $mapping['mapel_detail_id'],
                            'nilai_akhir' => $nilai,
                            'deskripsi' => $deskripsi,
                            'deleted' => '0',
                        ]);
                        Log::info("Created new detail: {$mapelName}");
                    }
                } else {
                    // Process head
                    $existingDetail = BookDetail::where('book_head_id', $bookHead->id)
                        ->where('mapel_head_id', $mapping['mapel_head_id'])
                        ->first();

                    if ($existingDetail) {
                        $existingDetail->update([
                            'nilai_akhir' => $nilai,
                            'deskripsi' => $deskripsi,
                        ]);
                        Log::info("Updated head: {$mapelName}");
                    } else {
                        BookDetail::create([
                            'book_head_id' => $bookHead->id,
                            'mapel_head_id' => $mapping['mapel_head_id'],
                            'mapel_detail_id' => null,
                            'nilai_akhir' => $nilai,
                            'deskripsi' => $deskripsi,
                            'deleted' => '0',
                        ]);
                        Log::info("Created new head: {$mapelName}");
                    }
                }
            } else {
                Log::info("Skipped {$mapping['type']}: {$mapelName} - no nilai or deskripsi");
            }
        }
    }

    public function getErrors()
    {
        return $this->errors;
    }

    public function getSuccessCount()
    {
        return $this->successCount;
    }
}
