<?php

namespace App\Http\Controllers;

use App\Models\Kelas;
use App\Models\PeriodeWaliKelas;
use App\Models\TahunAjaran;
use App\Models\WaliKelas;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class WaliKelasPeriodeController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $editId = $request->get('edit'); // ambil id dari query string

        $waliKelasPeriodeEdit = null;
        if ($editId) {
            $waliKelasPeriodeEdit = PeriodeWaliKelas::findOrFail($editId);
        }

        $perPage = $request->query('per_page', 20);
        $search = $request->input('search');

        // Get data for display
        $dataWaliKelasPeriode = PeriodeWaliKelas::with(['kelas', 'waliKelas'])
            ->when($search, function ($query) use ($search) {
                $query->WhereHas('kelas', function ($q) use ($search) {
                    $q->where('nama_kelas_nomor', 'like', '%' . $search . '%');
                })
                    ->orWhereHas('kelas', function ($q) use ($search) {
                        $q->where('nama_kelas_sub', 'like', '%' . $search . '%');
                    })
                    ->orWhereHas('waliKelas', function ($q) use ($search) {
                        $q->where('nama_guru', 'like', '%' . $search . '%');
                    })
                    // Tambahkan pencarian di kolom tabel utama
                    ->orWhere('thn_ajaran_awal', 'like', '%' . $search . '%')
                    ->orWhere('thn_ajaran_akhir', 'like', '%' . $search . '%');
            })
            ->join('kelas', 'periode_wali_kelas.kelas_id', '=', 'kelas.id')
            ->join('wali_kelas', 'periode_wali_kelas.wali_kelas_id', '=', 'wali_kelas.id')
            ->select('periode_wali_kelas.*')
            ->orderBy('periode_wali_kelas.thn_ajaran_akhir', 'desc')
            ->orderBy('kelas.nama_kelas_nomor', 'asc')
            ->orderBy('kelas.nama_kelas_sub', 'asc')
            ->orderBy('wali_kelas.nama_guru', 'asc')
            ->where('periode_wali_kelas.deleted', '0')
            ->paginate($perPage)
            ->appends(request()->query());

        $dataKelas = Kelas::where('deleted', '0')
            ->orderBy('nama_kelas_nomor', 'asc')
            ->orderBy('nama_kelas_sub', 'asc')
            ->get();
        $dataWaliKelas = WaliKelas::where('deleted', '0')->orderBy('nama_guru', 'asc')->get();
        
        // Get all TahunAjaran data, ordered by tahun_akhir descending
        $dataTahunAjaran = TahunAjaran::where('deleted', '0')
            ->orderBy('tahun_akhir', 'desc')
            ->get();

        return view('pages.wali_kelas_periode.index', compact('dataWaliKelasPeriode', 'dataKelas', 'dataWaliKelas', 'dataTahunAjaran', 'waliKelasPeriodeEdit'))
            ->with('i', (request()->input('page', 1) - 1) * $perPage);
    }
    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return redirect()->route('wali-kelas-periode.index');
    }

    /**
     * Store a newly created resource in storage.
     */

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'wali_kelas_id' => 'required',
            'kelas_id' => 'required',
            'tahun_ajaran' => 'required',
        ], [
            'wali_kelas_id.required' => 'Wali kelas harus diisi',
            'kelas_id.required' => 'Kelas harus diisi',
            'tahun_ajaran.required' => 'Tahun ajaran harus diisi',
        ]);

        $validateData = $validator->validate();

        // Pisahkan tahun_ajaran (misal "2025/2026") jadi awal & akhir
        [$tahunAwal, $tahunAkhir] = explode('/', $validateData['tahun_ajaran']);

        // 🔍 Cek apakah kombinasi kelas + tahun ajaran sudah ada
        $exists = PeriodeWaliKelas::where('kelas_id', $request->kelas_id)
            ->where('thn_ajaran_awal', $tahunAwal)
            ->where('thn_ajaran_akhir', $tahunAkhir)
            ->where('deleted', '0') // jika kamu pakai soft delete manual
            ->exists();

        if ($exists) {
            return back()
                ->withErrors([
                    'kelas_id' => 'Data tersebut sudah ada!',
                    'tahun_ajaran' => 'Data tersebut sudah ada!',
                ])
                ->withInput();
        }

        try {
            // Pisahkan tahun_ajaran (misal "2025/2026") jadi awal & akhir
            [$tahunAwal, $tahunAkhir] = explode('/', $validateData['tahun_ajaran']);

            PeriodeWaliKelas::create([
                'wali_kelas_id' => $validateData['wali_kelas_id'],
                'kelas_id' => $validateData['kelas_id'],
                'tahun_ajaran' => $validateData['tahun_ajaran'],
                'thn_ajaran_awal' => $tahunAwal,
                'thn_ajaran_akhir' => $tahunAkhir,
            ]);

            return redirect()->route('wali-kelas-periode.index')->with('successNotif', 'Data periode berhasil ditambahkan!');
        } catch (\Throwable $th) {
            Log::error('Error saat menyimpan data periode wali kelas: ' . $th->getMessage());
            return back()->with('error', 'Data periode wali kelas gagal ditambahkan')->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        return redirect()->route('wali-kelas-periode.index', ['edit' => $id]);
    }

    /**
     * Update the specified resource in storage.
     */

    public function update(Request $request, string $id)
    {
        $validator = Validator::make($request->all(), [
            'wali_kelas_id' => 'required',
            'kelas_id' => 'required',
            'tahun_ajaran' => 'required',
        ], [
            'wali_kelas_id.required' => 'Wali kelas harus diisi',
            'kelas_id.required' => 'Kelas harus diisi',
            'tahun_ajaran.required' => 'Tahun ajaran harus diisi',
        ]);

        $validateData = $validator->validate();

        // Pisahkan tahun_ajaran (misal "2025/2026") jadi awal & akhir
        [$tahunAwal, $tahunAkhir] = explode('/', $validateData['tahun_ajaran']);

        // 🔍 Cek apakah kombinasi kelas + tahun ajaran sudah ada (kecuali data yang sedang diupdate)
        $exists = PeriodeWaliKelas::where('kelas_id', $request->kelas_id)
            ->where('thn_ajaran_awal', $tahunAwal)
            ->where('thn_ajaran_akhir', $tahunAkhir)
            ->where('deleted', '0') // jika kamu pakai soft delete manual
            ->where('id', '!=', $id)
            ->exists();

        if ($exists) {
            return back()
                ->withErrors([
                    'kelas_id' => 'Data tersebut sudah ada!',
                    'tahun_ajaran' => 'Data tersebut sudah ada!',
                ])
                ->withInput();
        }

        try {
            $periode = PeriodeWaliKelas::findOrFail($id);

            $periode->update([
                'wali_kelas_id' => $validateData['wali_kelas_id'],
                'kelas_id' => $validateData['kelas_id'],
                'tahun_ajaran' => $validateData['tahun_ajaran'],
                'thn_ajaran_awal' => $tahunAwal,
                'thn_ajaran_akhir' => $tahunAkhir,
            ]);

            return redirect()->route('wali-kelas-periode.index')->with('successNotif', 'Data periode berhasil diperbarui!');
        } catch (\Throwable $th) {
            Log::error('Error saat memperbarui data periode wali kelas: ' . $th->getMessage());
            return back()->with('error', 'Data periode wali kelas gagal diperbarui')->withInput();
        }
    }


    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
}
