<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\WaliKelas;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;

class WaliKelasController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $perPage = $request->query('per_page', 20);
        $search = $request->input('search');

        $dataWaliKelas = WaliKelas::with(['user'])->when($search, function ($query) use ($search) {
            return $query->where('nama_guru', 'like', '%' . $search . '%')
                ->orWhere('nip', 'like', '%' . $search . '%')
                ->orWhere('no_hp', 'like', '%' . $search . '%');
        })
            ->where('deleted', '0')
            ->orderBy('nama_guru', 'asc')
            ->paginate($perPage)
            ->appends(request()->query());

        return view('pages.wali_kelas.index', compact('dataWaliKelas'))
            ->with('i', (request()->input('page', 1) - 1) * $perPage);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validateData = $request->validate([
            'nip' => 'nullable|unique:wali_kelas,nip|max:30',
            'nama_guru' => 'required',
            'jenis_kelamin' => 'required|in:Laki-Laki,Perempuan',
            'alamat' => 'nullable',
            'no_hp' => 'required|max:15|unique:wali_kelas,no_hp',
            'email' => 'nullable|email|unique:wali_kelas,email|max:50',
        ], [
            'nip.unique' => 'NIP sudah terdaftar',
            'nama_guru.required' => 'Nama guru harus diisi',
            'jenis_kelamin.required' => 'Jenis kelamin harus dipilih',
            'email.email' => 'Format email tidak valid',
            'email.unique' => 'Email sudah terdaftar',
            'no_hp.required' => 'No HP harus diisi',
            'no_hp.max' => 'No HP maksimal 15 karakter',
            'no_hp.unique' => 'No HP sudah terdaftar',
        ]);

        DB::beginTransaction();
        try {
            // Create user first
            $user = User::create([
                'name' => $validateData['nama_guru'],
                'email' => $validateData['no_hp'],
                'password' => Hash::make('password'),
                'role' => 'Wali',
            ]);

            // Create wali kelas
            WaliKelas::create([
                'user_id' => $user->id,
                'nip' => $validateData['nip'],
                'nama_guru' => $validateData['nama_guru'],
                'jenis_kelamin' => $validateData['jenis_kelamin'],
                'alamat' => $validateData['alamat'],
                'no_hp' => $validateData['no_hp'],
                'email' => $validateData['email'],
            ]);

            DB::commit();
            return back()->with('successNotif', 'Data wali kelas "' . $validateData['nama_guru'] . '" berhasil ditambahkan');
        } catch (\Throwable $th) {
            DB::rollback();
            Log::error('Error saat menyimpan data wali kelas: ' . $th->getMessage());
            return back()->with('error', 'Data wali kelas gagal ditambahkan');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $waliKelas = WaliKelas::findOrFail($id);

        $validateData = $request->validate([
            'nip' => 'nullable|unique:wali_kelas,nip,' . $id . '|max:30',
            'nama_guru' => 'required|max:50',
            'jenis_kelamin' => 'required|in:Laki-Laki,Perempuan',
            'alamat' => 'nullable',
            'no_hp' => 'required|max:15|unique:wali_kelas,no_hp,' . $id,
            'email' => 'nullable|email|unique:wali_kelas,email,' . $id,
            'status' => 'required|in:Aktif,Non-Aktif',
        ], [
            'nip.unique' => 'NIP sudah terdaftar',
            'nama_guru.required' => 'Nama guru harus diisi',
            'jenis_kelamin.required' => 'Jenis kelamin harus dipilih',
            'email.email' => 'Format email tidak valid',
            'email.unique' => 'Email sudah terdaftar',
            'status.required' => 'Status harus dipilih',
            'no_hp.required' => 'No HP harus diisi',
            'no_hp.max' => 'No HP maksimal 15 karakter',
            'no_hp.unique' => 'No HP sudah terdaftar',
        ]);

        DB::beginTransaction();
        try {
            // Update user
            $waliKelas->user->update([
                'name' => $validateData['nama_guru'],
                'email' => $validateData['no_hp'],
            ]);

            // Update wali kelas
            $waliKelas->update([
                'nip' => $validateData['nip'],
                'nama_guru' => $validateData['nama_guru'],
                'jenis_kelamin' => $validateData['jenis_kelamin'],
                'alamat' => $validateData['alamat'],
                'no_hp' => $validateData['no_hp'],
                'email' => $validateData['email'],
                'status' => $validateData['status'],
            ]);

            DB::commit();
            return back()->with('successNotif', 'Data wali kelas "' . $validateData['nama_guru'] . '" berhasil diperbarui');
        } catch (\Throwable $th) {
            DB::rollback();
            Log::error('Error saat memperbarui data wali kelas: ' . $th->getMessage());
            return back()->with('error', 'Data wali kelas gagal diperbarui');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }

    /**
     * Reset password wali kelas to default
     */
    public function resetPassword(string $id)
    {
        try {
            $waliKelas = WaliKelas::findOrFail($id);
            $user = User::findOrFail($waliKelas->user_id);

            // Reset password to default
            $user->password = Hash::make('Raport123!');
            $user->save();

            return redirect()->back()->with('successNotif', 'Password berhasil direset menjadi "Raport123!"');
        } catch (\Exception $e) {
            Log::error('Error resetting password: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Gagal mereset password: ' . $e->getMessage());
        }
    }
}
