<?php

namespace App\Http\Controllers;

use App\Models\TahunAjaran;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class TahunAjaranController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $perPage = $request->query('per_page', 20);
        $search = $request->input('search');

        // Get data for display
        $dataTahunAjaran = TahunAjaran::when($search, function ($query) use ($search) {
            return $query->where(function ($q) use ($search) {
                $q->where('tahun_awal', 'like', '%' . $search . '%')
                    ->orWhere('tahun_akhir', 'like', '%' . $search . '%');
            });
        })
            ->where('deleted', '0')
            ->orderBy('tahun_akhir', 'desc')
            ->paginate($perPage)
            ->appends(request()->query());


        // Get kelas for edit if edit parameter exists
        $tahunAjaranEdit = null;
        if ($request->has('edit')) {
            $tahunAjaranEdit = TahunAjaran::find($request->get('edit'));
        }

        return view('pages.tahun_ajaran.index', compact('dataTahunAjaran', 'tahunAjaranEdit'))
            ->with('i', (request()->input('page', 1) - 1) * $perPage);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return redirect()->route('tahun-ajaran.index');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validateData = $request->validate([
            'tahun_awal' => 'required|digits:4|integer|unique:tahun_ajarans,tahun_awal',
            'tahun_akhir' => 'required|digits:4|integer|unique:tahun_ajarans,tahun_akhir',
            'status' => 'required|in:Aktif,Tidak Aktif',
        ], [
            'tahun_awal.required' => 'Tahun awal harus diisi',
            'tahun_awal.digits' => 'Tahun awal harus 4 angka',
            'tahun_awal.integer' => 'Tahun awal harus berupa angka',
            'tahun_awal.unique' => 'Tahun awal sudah terdaftar',
            'tahun_akhir.required' => 'Tahun akhir harus diisi',
            'tahun_akhir.digits' => 'Tahun akhir harus 4 angka',
            'tahun_akhir.integer' => 'Tahun akhir harus berupa angka',
            'tahun_akhir.unique' => 'Tahun akhir sudah terdaftar',
            'status.required' => 'Status harus diisi',
            'status.in' => 'Status harus Aktif atau Tidak Aktif',
        ]);

        // ✅ Validasi tambahan manual
        if ($validateData['tahun_awal'] > $validateData['tahun_akhir']) {
            return back()
                ->withErrors(['tahun_awal' => 'Tahun awal tidak boleh lebih besar dari tahun akhir'])
                ->withInput();
        }

        if (($validateData['tahun_akhir'] - $validateData['tahun_awal']) !== 1) {
            return back()
                ->withErrors(['tahun_akhir' => 'Selisih antara tahun awal dan tahun akhir harus 1 tahun'])
                ->withInput();
        }

        try {
            TahunAjaran::create([
                'tahun_awal' => $validateData['tahun_awal'],
                'tahun_akhir' => $validateData['tahun_akhir'],
                'status' => $validateData['status'],
                'deleted' => '0',
            ]);
            return redirect()->route('tahun-ajaran.index')->with('successNotif', 'Data tahun ajaran "' . $validateData['tahun_awal'] . '/' . $validateData['tahun_akhir'] . '" berhasil ditambahkan');
        } catch (\Throwable $th) {
            Log::error('Error saat menyimpan data tahun ajaran: ' . $th->getMessage());
            return back()->with('error', 'Data tahun ajaran gagal ditambahkan')->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        return redirect()->route('tahun-ajaran.index', ['edit' => $id]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $validateData = $request->validate([
            'tahun_awal' => 'required|digits:4|integer|unique:tahun_ajarans,tahun_awal,' . $id,
            'tahun_akhir' => 'required|digits:4|integer|unique:tahun_ajarans,tahun_akhir,' . $id,
            'status' => 'required|in:Aktif,Tidak Aktif',
        ], [
            'tahun_awal.required' => 'Tahun awal harus diisi',
            'tahun_awal.digits' => 'Tahun awal harus 4 angka',
            'tahun_awal.integer' => 'Tahun awal harus berupa angka',
            'tahun_awal.unique' => 'Tahun awal sudah terdaftar',
            'tahun_akhir.required' => 'Tahun akhir harus diisi',
            'tahun_akhir.digits' => 'Tahun akhir harus 4 angka',
            'tahun_akhir.integer' => 'Tahun akhir harus berupa angka',
            'tahun_akhir.unique' => 'Tahun akhir sudah terdaftar',
            'status.required' => 'Status harus diisi',
            'status.in' => 'Status harus Aktif atau Tidak Aktif',
        ]);

        // ✅ Validasi tambahan manual
        if ($validateData['tahun_awal'] > $validateData['tahun_akhir']) {
            return back()
                ->withErrors(['tahun_awal' => 'Tahun awal tidak boleh lebih besar dari tahun akhir'])
                ->withInput();
        }

        if (($validateData['tahun_akhir'] - $validateData['tahun_awal']) !== 1) {
            return back()
                ->withErrors(['tahun_akhir' => 'Selisih antara tahun awal dan tahun akhir harus 1 tahun'])
                ->withInput();
        }

        try {
            $tahunAjaran = TahunAjaran::findOrFail($id);
            $tahunAjaran->update([
                'tahun_awal' => $validateData['tahun_awal'],
                'tahun_akhir' => $validateData['tahun_akhir'],
                'status' => $validateData['status'],
            ]);
            return redirect()->route('tahun-ajaran.index')->with('successNotif', 'Data tahun ajaran "' . $validateData['tahun_awal'] . '/' . $validateData['tahun_akhir'] . '" berhasil diperbarui');
        } catch (\Throwable $th) {
            Log::error('Error saat memperbarui data tahun ajaran: ' . $th->getMessage());
            return back()->with('error', 'Data tahun ajaran gagal diperbarui')->withInput();
        }
    }

    /**
     * Check if tahun ajaran is being used
     */
    public function checkUsage(string $id)
    {
        try {
            $tahunAjaran = TahunAjaran::findOrFail($id);
            
            // Check if used in siswas table
            $usedInSiswas = \App\Models\Siswa::where('thn_ajaran_awal', $tahunAjaran->tahun_awal)
                ->where('thn_ajaran_akhir', $tahunAjaran->tahun_akhir)
                ->where('deleted', '0')
                ->exists();
            
            // Check if used in book_heads table
            $usedInBookHeads = \App\Models\BookHead::where('thn_ajaran_awal', $tahunAjaran->tahun_awal)
                ->where('thn_ajaran_akhir', $tahunAjaran->tahun_akhir)
                ->where('deleted', '0')
                ->exists();
            
            return response()->json([
                'used' => $usedInSiswas || $usedInBookHeads,
                'tahun_ajaran' => $tahunAjaran->tahun_awal . '/' . $tahunAjaran->tahun_akhir
            ]);
        } catch (\Throwable $th) {
            return response()->json([
                'error' => 'Terjadi kesalahan saat memeriksa data'
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        try {
            $tahunAjaran = TahunAjaran::findOrFail($id);
            
            // Check if used in siswas or book_heads
            $usedInSiswas = \App\Models\Siswa::where('thn_ajaran_awal', $tahunAjaran->tahun_awal)
                ->where('thn_ajaran_akhir', $tahunAjaran->tahun_akhir)
                ->where('deleted', '0')
                ->exists();
            
            $usedInBookHeads = \App\Models\BookHead::where('thn_ajaran_awal', $tahunAjaran->tahun_awal)
                ->where('thn_ajaran_akhir', $tahunAjaran->tahun_akhir)
                ->where('deleted', '0')
                ->exists();
            
            if ($usedInSiswas || $usedInBookHeads) {
                return back()->with('error', 'Data tahun ajaran "' . $tahunAjaran->tahun_awal . '/' . $tahunAjaran->tahun_akhir . '" tidak dapat dihapus karena sudah digunakan');
            }
            
            // Soft delete
            $tahunAjaran->update(['deleted' => '1']);
            
            return redirect()->route('tahun-ajaran.index')->with('successNotif', 'Data tahun ajaran "' . $tahunAjaran->tahun_awal . '/' . $tahunAjaran->tahun_akhir . '" berhasil dihapus');
        } catch (\Throwable $th) {
            Log::error('Error saat menghapus data tahun ajaran: ' . $th->getMessage());
            return back()->with('error', 'Data tahun ajaran gagal dihapus');
        }
    }
}
