<?php

namespace App\Http\Controllers;

use App\Models\Siswa;
use App\Models\Kelas;
use App\Models\Pekerjaan;
use App\Models\PeriodeWaliKelas;
use App\Models\TahunAjaran;
use App\Models\WaliKelas;
use App\Models\TemporarySiswa;
use App\Models\RaportHead;
use App\Models\BookHead;
use App\Models\BookDetail;
use App\Models\MapelHead;
use App\Models\MapelKelas;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Maatwebsite\Excel\Facades\Excel;
use App\Imports\SiswaImport;
use App\Exports\SiswaExport;

class SiswaController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $perPage = $request->query('per_page', 20);
        $search = $request->input('searchNamaSiswa');
        $filterJenisKelamin = $request->input('selectJenisKelamin');
        $filterKelas = $request->input('selectKelas');
        $filterAngkatan = $request->input('selectAngkatan');
        $filterStatus = $request->input('selectStatus');
        $filterWaliKelas = $request->input('waliKelasSelect');
        $filterTahunAjaran = $request->input('tahunAjaranSelect');

        $dataSiswa = Siswa::leftJoin('kelas', 'siswas.kelas_id', '=', 'kelas.id')
            ->leftJoin('periode_wali_kelas', function ($join) {
                $join->on('siswas.kelas_id', '=', 'periode_wali_kelas.kelas_id')
                    ->on('siswas.thn_ajaran_awal', '=', 'periode_wali_kelas.thn_ajaran_awal')
                    ->on('siswas.thn_ajaran_akhir', '=', 'periode_wali_kelas.thn_ajaran_akhir');
            })
            ->leftJoin('wali_kelas', 'periode_wali_kelas.wali_kelas_id', '=', 'wali_kelas.id')
            ->select(
                'siswas.*',
                'kelas.nama_kelas_nomor',
                'kelas.nama_kelas_sub',
                'wali_kelas.nama_guru as nama_wali_kelas'
            )
            ->with(['kelas', 'pekerjaanAyah', 'pekerjaanIbu'])
            ->when($search, function ($query) use ($search) {
                return $query->where('siswas.nama_siswa', 'like', '%' . $search . '%')
                    ->orWhere('siswas.nis', 'like', '%' . $search . '%')
                    ->orWhere('siswas.nisn', 'like', '%' . $search . '%');
            })
            ->when($filterJenisKelamin, function ($query) use ($filterJenisKelamin) {
                return $query->where('siswas.jenis_kelamin', $filterJenisKelamin);
            })
            ->when($filterKelas, function ($query) use ($filterKelas) {
                return $query->where('siswas.kelas_id', $filterKelas);
            })
            ->when($filterAngkatan, function ($query) use ($filterAngkatan) {
                return $query->where('siswas.angkatan', $filterAngkatan);
            })
            ->when($filterStatus, function ($query) use ($filterStatus) {
                return $query->where('siswas.status', $filterStatus);
            })
            ->when($filterWaliKelas, function ($query) use ($filterWaliKelas) {
                return $query->where('periode_wali_kelas.wali_kelas_id', $filterWaliKelas);
            })
            ->when($filterTahunAjaran, function ($query) use ($filterTahunAjaran) {
                return $query->where('siswas.thn_ajaran_awal', explode('/', $filterTahunAjaran)[0])
                    ->where('siswas.thn_ajaran_akhir', explode('/', $filterTahunAjaran)[1]);
            })
            ->where('siswas.deleted', '0')
            ->orderBy('siswas.thn_ajaran_akhir', 'desc')
            ->orderBy('siswas.angkatan', 'desc')
            ->orderBy('kelas.nama_kelas_nomor', 'asc')
            ->orderBy('kelas.nama_kelas_sub', 'asc')
            ->orderBy('siswas.nama_siswa', 'asc')
            ->paginate($perPage)
            ->appends(request()->query());

        // Get filter options
        $jenisKelaminOptions = Siswa::where('deleted', '0')->distinct()->pluck('jenis_kelamin');
        $kelasOptions = Kelas::where('deleted', '0')
            ->orderBy('nama_kelas_nomor', 'asc')
            ->orderBy('nama_kelas_sub', 'asc')
            ->get();
        $angkatanOptions = Siswa::where('deleted', '0')->distinct()->orderBy('angkatan', 'desc')->pluck('angkatan');
        $statusOptions = Siswa::where('deleted', '0')
            ->distinct()
            ->pluck('status')
            ->sortBy(function ($status) {
                return match ($status) {
                    'Aktif' => 1,
                    'Non-Aktif' => 2,
                    'Alumni' => 3,
                    default => 4,
                };
            })
            ->values();
        $waliKelasOptions = WaliKelas::whereHas('periodeWaliKelas', function ($query) {
            $query->where('deleted', '0');
        })
            ->select('wali_kelas.id', 'wali_kelas.nama_guru')
            ->distinct()
            ->orderBy('wali_kelas.nama_guru', 'asc')
            ->get();

        // Get the highest academic year from siswas table
        $highestTahunAjaran = Siswa::where('deleted', '0')
            ->where('status', 'Aktif')
            ->orderBy('thn_ajaran_akhir', 'desc')
            ->first();

        $currentThnAwal = $highestTahunAjaran ? $highestTahunAjaran->thn_ajaran_awal : (int) date('Y');
        $currentThnAkhir = $highestTahunAjaran ? $highestTahunAjaran->thn_ajaran_akhir : (int) date('Y') + 1;

        $activeStudentsCurrentYear = Siswa::where('deleted', '0')
            ->where('status', 'Aktif')
            ->where('thn_ajaran_awal', $currentThnAwal)
            ->where('thn_ajaran_akhir', $currentThnAkhir)
            ->get();

        // Check if all active students have completed input for both semesters
        $canTriggerPromotion = $this->checkAllStudentsInputComplete($activeStudentsCurrentYear, $currentThnAwal, $currentThnAkhir);

        $hasTemporaryData = TemporarySiswa::query()->exists();

        return view('pages.siswa.index', compact('dataSiswa', 'jenisKelaminOptions', 'kelasOptions', 'angkatanOptions', 'statusOptions', 'waliKelasOptions', 'canTriggerPromotion', 'hasTemporaryData'))
            ->with('i', (request()->input('page', 1) - 1) * $perPage)
            ->with('selectedTahunAjaran', $filterTahunAjaran);
    }

    /**
     * Check if all students have completed input (Sudah Input) for both semesters
     */
    private function checkAllStudentsInputComplete($students, $thnAjaranAwal, $thnAjaranAkhir): bool
    {
        if ($students->isEmpty()) {
            return false;
        }

        foreach ($students as $siswa) {
            // Check for both semesters
            foreach (['Ganjil', 'Genap'] as $semester) {
                // Get BookHead for this student and semester
                $bookHead = BookHead::where('siswa_id', $siswa->id)
                    ->where('thn_ajaran_awal', $thnAjaranAwal)
                    ->where('thn_ajaran_akhir', $thnAjaranAkhir)
                    ->where('semester', $semester)
                    ->where('deleted', '0')
                    ->with('details')
                    ->first();

                // If no BookHead, status is "Belum Input"
                if (!$bookHead) {
                    return false;
                }

                // Get mapel assigned to this student's kelas
                $mapelAssignments = MapelKelas::where('kelas_id', $siswa->kelas_id)->get();

                if ($mapelAssignments->isEmpty()) {
                    continue; // Skip if no mapel assigned
                }

                // Separate mapel heads and details
                $mapelHeadIds = $mapelAssignments->where('mapel_type', 'head')->pluck('mapel_id')->toArray();
                $mapelDetailIds = $mapelAssignments->where('mapel_type', 'detail')->pluck('mapel_id')->toArray();

                // Get MapelHead with their details
                $mapelHeads = MapelHead::with(['details' => function ($query) use ($mapelDetailIds) {
                    $query->where('status', 'Aktif')
                        ->where('deleted', '0')
                        ->whereIn('id', $mapelDetailIds);
                }])
                    ->where('status', 'Aktif')
                    ->where('deleted', '0')
                    ->whereIn('id', $mapelHeadIds)
                    ->get();

                if ($mapelHeads->isEmpty()) {
                    continue; // Skip if no active mapel
                }

                // Build bookDetails map for quick lookup
                $bookDetails = $bookHead->details->keyBy(function ($detail) {
                    return $detail->mapel_detail_id ?? 'head_' . $detail->mapel_head_id;
                });

                // Check if all mapel have complete input (nilai_akhir and deskripsi)
                foreach ($mapelHeads as $mapelHead) {
                    if ($mapelHead->details->isNotEmpty()) {
                        // Mapel has details
                        foreach ($mapelHead->details as $detail) {
                            $detailData = $bookDetails->get($detail->id);

                            // Check if detail exists and has both nilai_akhir and deskripsi
                            if (!$detailData || empty($detailData->nilai_akhir) || empty($detailData->deskripsi)) {
                                return false; // Status "Belum Input"
                            }
                        }
                    } else {
                        // Mapel has no details, check head directly
                        $headData = $bookDetails->get('head_' . $mapelHead->id);

                        // Check if head exists and has both nilai_akhir and deskripsi
                        if (!$headData || empty($headData->nilai_akhir) || empty($headData->deskripsi)) {
                            return false; // Status "Belum Input"
                        }
                    }
                }
            }
        }

        // All students have complete input for both semesters
        return true;
    }

    public function riwayat(Request $request)
    {
        if (!TemporarySiswa::query()->exists()) {
            return redirect()->route('siswa.index')->with('error', 'Riwayat siswa tidak tersedia.');
        }

        $perPage = $request->query('per_page', 20);
        $search = $request->input('searchNamaSiswa');
        $filterJenisKelamin = $request->input('selectJenisKelamin');
        $filterKelas = $request->input('selectKelas');
        $filterAngkatan = $request->input('selectAngkatan');
        $filterStatus = $request->input('selectStatus');
        $filterWaliKelas = $request->input('waliKelasSelect');
        $filterStatusKenaikan = $request->input('selectStatusKenaikan');

        $riwayatSiswa = TemporarySiswa::with(['siswa.kelas', 'kelas'])
            ->leftJoin('siswas', 'temporary_siswas.siswa_id', '=', 'siswas.id')
            ->leftJoin('kelas', 'temporary_siswas.kelas_id', '=', 'kelas.id')
            ->leftJoin('periode_wali_kelas', function ($join) {
                $join->on('temporary_siswas.kelas_id', '=', 'periode_wali_kelas.kelas_id')
                    ->on('temporary_siswas.thn_ajaran_awal', '=', 'periode_wali_kelas.thn_ajaran_awal')
                    ->on('temporary_siswas.thn_ajaran_akhir', '=', 'periode_wali_kelas.thn_ajaran_akhir');
            })
            ->leftJoin('wali_kelas', 'periode_wali_kelas.wali_kelas_id', '=', 'wali_kelas.id')
            ->select('temporary_siswas.*', 'wali_kelas.nama_guru as nama_wali_kelas')
            ->when($search, function ($query) use ($search) {
                return $query->where(function ($q) use ($search) {
                    $q->where('siswas.nama_siswa', 'like', '%' . $search . '%')
                        ->orWhere('siswas.nis', 'like', '%' . $search . '%')
                        ->orWhere('siswas.nisn', 'like', '%' . $search . '%');
                });
            })
            ->when($filterJenisKelamin, function ($query) use ($filterJenisKelamin) {
                return $query->where('siswas.jenis_kelamin', $filterJenisKelamin);
            })
            ->when($filterKelas, function ($query) use ($filterKelas) {
                return $query->where('temporary_siswas.kelas_id', $filterKelas);
            })
            ->when($filterAngkatan, function ($query) use ($filterAngkatan) {
                return $query->where('siswas.angkatan', $filterAngkatan);
            })
            ->when($filterStatus, function ($query) use ($filterStatus) {
                return $query->where('temporary_siswas.status', $filterStatus);
            })
            ->when($filterWaliKelas, function ($query) use ($filterWaliKelas) {
                return $query->where('periode_wali_kelas.wali_kelas_id', $filterWaliKelas);
            })
            ->when($filterStatusKenaikan, function ($query) use ($filterStatusKenaikan) {
                return $query->where(function ($statusQuery) use ($filterStatusKenaikan) {
                    if ($filterStatusKenaikan === 'Naik Kelas') {
                        $statusQuery->whereNull('temporary_siswas.keterangan_kenaikan')
                            ->whereIn('kelas.nama_kelas_nomor', [7, 8]);
                    } elseif ($filterStatusKenaikan === 'Lulus') {
                        $statusQuery->whereNull('temporary_siswas.keterangan_kenaikan')
                            ->where('kelas.nama_kelas_nomor', 9);
                    } else {
                        $statusQuery->where('temporary_siswas.keterangan_kenaikan', $filterStatusKenaikan);
                    }
                });
            })
            ->orderBy('siswas.thn_ajaran_akhir', 'desc')
            ->orderBy('siswas.angkatan', 'desc')
            ->orderBy('kelas.nama_kelas_nomor', 'asc')
            ->orderBy('kelas.nama_kelas_sub', 'asc')
            ->orderBy('siswas.nama_siswa', 'asc')
            ->paginate($perPage)
            ->appends($request->query());

        // Get the latest saved date
        $latestRiwayat = TemporarySiswa::orderBy('created_at', 'desc')->first();

        $jenisKelaminOptions = Siswa::where('deleted', '0')->distinct()->pluck('jenis_kelamin');
        $kelasOptions = Kelas::where('deleted', '0')
            ->orderBy('nama_kelas_nomor', 'asc')
            ->orderBy('nama_kelas_sub', 'asc')
            ->get();
        $angkatanOptions = Siswa::where('deleted', '0')->distinct()->orderBy('angkatan', 'desc')->pluck('angkatan');
        $statusOptions = Siswa::where('deleted', '0')
            ->distinct()
            ->pluck('status')
            ->sortBy(function ($status) {
                return match ($status) {
                    'Aktif' => 1,
                    'Non-Aktif' => 2,
                    'Alumni' => 3,
                    default => 4,
                };
            })
            ->values();

        $statusKenaikanOptions = ['Naik Kelas', 'Tidak Naik Kelas', 'Lulus', 'Tidak Lulus'];

        $waliKelasOptions = WaliKelas::whereHas('periodeWaliKelas', function ($query) {
            $query->where('deleted', '0');
        })
            ->select('wali_kelas.id', 'wali_kelas.nama_guru')
            ->distinct()
            ->orderBy('wali_kelas.nama_guru', 'asc')
            ->get();

        // Check if any student has raport data in the next academic year
        $hasRaportConflict = false;
        if ($latestRiwayat) {
            $nextThnAwal = $latestRiwayat->thn_ajaran_awal + 1;
            $nextThnAkhir = $latestRiwayat->thn_ajaran_akhir + 1;

            $hasRaportConflict = BookHead::whereIn('siswa_id', function ($query) {
                $query->select('siswa_id')
                    ->from('temporary_siswas')
                    ->distinct();
            })
                ->where('thn_ajaran_awal', $nextThnAwal)
                ->where('thn_ajaran_akhir', $nextThnAkhir)
                ->where('deleted', '0')
                ->exists();
        }

        return view('pages.siswa.riwayat', compact('riwayatSiswa', 'jenisKelaminOptions', 'kelasOptions', 'angkatanOptions', 'statusOptions', 'waliKelasOptions', 'latestRiwayat', 'statusKenaikanOptions', 'hasRaportConflict'))
            ->with('i', (request()->input('page', 1) - 1) * $perPage)
            ->with('perPage', $perPage)
            ->with('searchNamaSiswa', $search);
    }

    public function promote(Request $request)
    {
        try {
            // Delete all existing records from temporary_siswas table
            TemporarySiswa::truncate();

            // Get the highest academic year for active students
            $highestTahunAjaran = Siswa::where('deleted', '0')
                ->where('status', 'Aktif')
                ->orderBy('thn_ajaran_akhir', 'desc')
                ->first();

            if (!$highestTahunAjaran) {
                return redirect()->route('siswa.index')->with('error', 'Tidak ada siswa aktif yang ditemukan.');
            }

            $currentThnAwal = $highestTahunAjaran->thn_ajaran_awal;
            $currentThnAkhir = $highestTahunAjaran->thn_ajaran_akhir;

            // Only get active students in the current (highest) academic year
            $students = Siswa::where('deleted', '0')
                ->where('status', 'Aktif')
                ->where('thn_ajaran_awal', $currentThnAwal)
                ->where('thn_ajaran_akhir', $currentThnAkhir)
                ->with('kelas')
                ->get();

            if ($students->isEmpty()) {
                return redirect()->route('siswa.index')->with('error', 'Tidak ada siswa aktif pada tahun ajaran saat ini.');
            }

            $promotedCount = 0;
            $graduatedCount = 0;

            DB::transaction(function () use ($students, &$promotedCount, &$graduatedCount) {

                $timestamp = now();

                $temporaryPayload = $students->map(function ($siswa) use ($timestamp) {
                    return [
                        'kelas_id' => $siswa->kelas_id,
                        'siswa_id' => $siswa->id,
                        'thn_ajaran_awal' => $siswa->thn_ajaran_awal,
                        'thn_ajaran_akhir' => $siswa->thn_ajaran_akhir,
                        'status' => $siswa->status,
                        'keterangan_kenaikan' => $siswa->keterangan_kenaikan,
                        'created_at' => $timestamp,
                        'updated_at' => $timestamp,
                    ];
                })->all();

                TemporarySiswa::insert($temporaryPayload);

                $kelasLookup = Kelas::where('deleted', '0')
                    ->get()
                    ->groupBy(function ($kelas) {
                        return $kelas->nama_kelas_nomor . '|' . ($kelas->nama_kelas_sub ?? '');
                    });

                foreach ($students as $siswa) {
                    $kelas = $siswa->kelas;
                    if (!$kelas) {
                        continue;
                    }

                    $kelasNomor = (int) $kelas->nama_kelas_nomor;

                    // Check if student has keterangan_kenaikan
                    if (!is_null($siswa->keterangan_kenaikan)) {
                        // Only process students with "Tidak Naik Kelas" or "Tidak Lulus"
                        if (in_array($siswa->keterangan_kenaikan, ['Tidak Naik Kelas', 'Tidak Lulus'])) {
                            $newThnAwal = $siswa->thn_ajaran_awal + 1;
                            $newThnAkhir = $siswa->thn_ajaran_akhir + 1;

                            $siswa->update([
                                'thn_ajaran_awal' => $newThnAwal,
                                'thn_ajaran_akhir' => $newThnAkhir,
                                'tahun_ajaran' => $newThnAwal . '/' . $newThnAkhir,
                            ]);

                            $promotedCount++;
                        }

                        continue;
                    }

                    // Students in class 9 become alumni
                    if ($kelasNomor === 9) {
                        $newThnAwal = $siswa->thn_ajaran_awal + 1;
                        $newThnAkhir = $siswa->thn_ajaran_akhir + 1;

                        $siswa->update([
                            'status' => 'Alumni',
                            'thn_ajaran_awal' => $newThnAwal,
                            'thn_ajaran_akhir' => $newThnAkhir,
                            'tahun_ajaran' => $newThnAwal . '/' . $newThnAkhir,
                        ]);

                        $graduatedCount++;
                        continue;
                    }

                    // Students in class 7 or 8 move to next class
                    $newKey = ($kelasNomor + 1) . '|' . ($kelas->nama_kelas_sub ?? '');
                    $targetGroup = $kelasLookup->get($newKey);

                    if (!$targetGroup) {
                        throw new \RuntimeException('Kelas tujuan tidak ditemukan untuk kelas ' . $kelas->nama_kelas_nomor . ($kelas->nama_kelas_sub ? ' ' . $kelas->nama_kelas_sub : '') . '.');
                    }

                    $targetKelas = $targetGroup->first();

                    $newThnAwal = $siswa->thn_ajaran_awal + 1;
                    $newThnAkhir = $siswa->thn_ajaran_akhir + 1;

                    $siswa->update([
                        'kelas_id' => $targetKelas->id,
                        'thn_ajaran_awal' => $newThnAwal,
                        'thn_ajaran_akhir' => $newThnAkhir,
                        'tahun_ajaran' => $newThnAwal . '/' . $newThnAkhir,
                    ]);

                    $promotedCount++;
                }
            });

            // Build success message
            if ($promotedCount === 0 && $graduatedCount === 0) {
                return redirect()->route('siswa.index')->with('error', 'Tidak ada siswa yang dapat diproses untuk kenaikan kelas.');
            }

            $message = 'Kenaikan kelas berhasil!';
            if ($promotedCount > 0) {
                $message .= ' ' . $promotedCount . ' siswa naik kelas.';
            }
            if ($graduatedCount > 0) {
                $message .= ' ' . $graduatedCount . ' siswa menjadi alumni.';
            }

            return redirect()->route('siswa.index')->with('successNotif', $message);
        } catch (\Throwable $th) {
            Log::error('Error saat melakukan kenaikan kelas: ' . $th->getMessage());
            Log::error('Stack trace: ' . $th->getTraceAsString());

            return redirect()->route('siswa.index')->with('error', 'Proses kenaikan kelas gagal: ' . $th->getMessage());
        }
    }

    public function restore(Request $request)
    {
        try {
            // Get temporary data and validate BEFORE transaction
            $temporaryData = TemporarySiswa::with('siswa')->get();

            if ($temporaryData->isEmpty()) {
                return redirect('/siswa')->with('error', 'Tidak ada data riwayat untuk dipulihkan.');
            }

            // Transaction only for update operations
            DB::transaction(function () use ($temporaryData) {
                foreach ($temporaryData as $temp) {
                    if ($temp->siswa) {
                        $temp->siswa->update([
                            'kelas_id' => $temp->kelas_id,
                            'thn_ajaran_awal' => $temp->thn_ajaran_awal,
                            'thn_ajaran_akhir' => $temp->thn_ajaran_akhir,
                            'tahun_ajaran' => $temp->thn_ajaran_awal . '/' . $temp->thn_ajaran_akhir,
                            'status' => $temp->status,
                            'keterangan_kenaikan' => $temp->keterangan_kenaikan,
                        ]);
                    }
                }

                // Delete all temporary data after restoration
                TemporarySiswa::truncate();
            });

            return redirect('/siswa')->with('successNotif', 'Data siswa berhasil dipulihkan ke kondisi sebelumnya.');
        } catch (\Throwable $th) {
            Log::error('Error saat memulihkan data siswa: ' . $th->getMessage());
            Log::error('Stack trace: ' . $th->getTraceAsString());

            return redirect('/siswa')->with('successNotif', 'Data siswa berhasil dipulihkan ke kondisi sebelumnya.');
        }
    }

    public function import(Request $request)
    {
        $request->validate([
            'file' => 'required|mimes:xlsx,xls|max:10240',
        ], [
            'file.required' => 'File Excel harus dipilih',
            'file.mimes' => 'File harus berformat Excel (.xlsx atau .xls)',
            'file.max' => 'Ukuran file maksimal 10MB',
        ]);

        try {
            $file = $request->file('file');

            Log::info('Starting siswa import from file: ' . $file->getClientOriginalName());

            $import = new SiswaImport();
            Excel::import($import, $file);

            $successCount = $import->getSuccessCount();
            $updateCount = $import->getUpdateCount();
            $errors = $import->getErrors();

            if (!empty($errors)) {
                $errorMessage = 'Import selesai dengan beberapa error. ';

                if ($successCount > 0) {
                    $errorMessage .= $successCount . ' siswa berhasil ditambahkan. ';
                }

                if ($updateCount > 0) {
                    $errorMessage .= $updateCount . ' siswa berhasil diperbarui. ';
                }

                return redirect()->route('siswa.index')
                    ->with('warningNotif', $errorMessage)
                    ->with('importErrors', $errors);
            }

            if ($successCount > 0 || $updateCount > 0) {
                $successMessage = 'Import siswa berhasil! ';

                if ($successCount > 0) {
                    $successMessage .= $successCount . ' siswa berhasil ditambahkan. ';
                }

                if ($updateCount > 0) {
                    $successMessage .= $updateCount . ' siswa berhasil diperbarui.';
                }

                return redirect()->route('siswa.index')->with('successNotif', $successMessage);
            }

            return redirect()->route('siswa.index')->with('errorNotif', 'Tidak ada data siswa yang berhasil diimport.');
        } catch (\Throwable $th) {
            Log::error('Error saat import siswa: ' . $th->getMessage());
            Log::error('Stack trace: ' . $th->getTraceAsString());
            return redirect()->route('siswa.index')->with('errorNotif', 'Gagal import siswa: ' . $th->getMessage());
        }
    }

    /**
     * Export siswa data to Excel
     */
    public function exportExcel()
    {
        try {
            $timestamp = now()->format('Y-m-d_His');
            $filename = "Data_Siswa_{$timestamp}.xlsx";
            
            return Excel::download(new SiswaExport(), $filename);
        } catch (\Exception $e) {
            Log::error('Error export siswa: ' . $e->getMessage());
            return back()->with('error', 'Terjadi kesalahan saat mengekspor data: ' . $e->getMessage());
        }
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $kelas = Kelas::where('deleted', '0')
            ->orderBy('nama_kelas_nomor', 'asc')
            ->orderBy('nama_kelas_sub', 'asc')
            ->get();
        $pekerjaan = Pekerjaan::where('deleted', '0')->orderBy('nama_pekerjaan', 'asc')->get();
        $tahunAjaran = TahunAjaran::where('deleted', '0')->orderBy('tahun_akhir', 'desc')->get();

        return view('pages.siswa.create', compact('kelas', 'pekerjaan', 'tahunAjaran'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validateData = $request->validate([
            'foto' => 'nullable|image|mimes:jpg,jpeg,png|max:2048',
            'nis' => 'required|max:50|unique:siswas,nis',
            'nisn' => 'required|max:50|unique:siswas,nisn',
            'no_kip' => 'nullable|max:50|unique:siswas,no_kip',
            'no_kk' => 'nullable|max:50',
            'nama_siswa' => 'required|max:50',
            'jenis_kelamin' => 'required|in:Laki-Laki,Perempuan',
            'tempat_lahir' => 'nullable|max:50',
            'tanggal_lahir' => 'nullable|date',
            'agama' => 'required|in:Islam,Kristen,Hindu,Buddha,Konghucu',
            'alamat' => 'nullable',
            'no_hp_siswa' => 'nullable|max:15',
            'nama_ayah' => 'nullable|max:50',
            'pekerjaan_ayah_id' => 'nullable|exists:pekerjaans,id',
            'nama_ibu' => 'nullable|max:50',
            'pekerjaan_ibu_id' => 'nullable|exists:pekerjaans,id',
            'no_hp_ortu' => 'nullable|max:15',
            'kelas_id' => 'required|exists:kelas,id',
            'angkatan' => 'required',
            'status' => 'required|in:Aktif,Non-Aktif',
            'tahun_ajaran' => 'required',
        ], [
            'nis.required' => 'NIS harus diisi',
            'nis.unique' => 'NIS sudah terdaftar',
            'nisn.required' => 'NISN harus diisi',
            'nisn.unique' => 'NISN sudah terdaftar',
            'no_kip.unique' => 'No KIP sudah terdaftar',
            'no_kip.max' => 'No KIP maksimal 50 karakter',
            'no_kk.max' => 'No KK maksimal 50 karakter',
            'nama_siswa.required' => 'Nama siswa harus diisi',
            'jenis_kelamin.required' => 'Jenis kelamin harus dipilih',
            'agama.required' => 'Agama harus dipilih',
            'kelas_id.required' => 'Kelas harus dipilih',
            'angkatan.required' => 'Angkatan harus diisi',
            'status.required' => 'Status harus dipilih',
            'tahun_ajaran.required' => 'Tahun ajaran harus dipilih',
        ]);

        // Pisahkan tahun_ajaran (misal "2025/2026") jadi awal & akhir
        [$tahunAwal, $tahunAkhir] = explode('/', $validateData['tahun_ajaran']);

        try {
            // Pisahkan tahun_ajaran (misal "2025/2026") jadi awal & akhir
            [$tahunAwal, $tahunAkhir] = explode('/', $validateData['tahun_ajaran']);
            // Handle file upload
            $foto = null;
            if ($request->hasFile('foto')) {
                $file = $request->file('foto');
                $nama_siswa_cleaned = str_replace([' ', '/'], '_', strtolower($validateData['nama_siswa']));
                $filename = $validateData['nisn'] . '_' . $nama_siswa_cleaned . '.' . $file->getClientOriginalExtension();
                $file->move(public_path('foto_siswa'), $filename);
                $foto = $filename;
            }

            Siswa::create([
                'foto' => $foto,
                'nis' => $validateData['nis'],
                'nisn' => $validateData['nisn'],
                'no_kip' => $validateData['no_kip'] ?? null,
                'no_kk' => $validateData['no_kk'] ?? null,
                'nama_siswa' => $validateData['nama_siswa'],
                'jenis_kelamin' => $validateData['jenis_kelamin'],
                'tempat_lahir' => $validateData['tempat_lahir'] ?? null,
                'tanggal_lahir' => $validateData['tanggal_lahir'] ?? null,
                'agama' => $validateData['agama'],
                'alamat' => $validateData['alamat'] ?? null,
                'no_hp_siswa' => $validateData['no_hp_siswa'] ?? null,
                'nama_ayah' => $validateData['nama_ayah'] ?? null,
                'pekerjaan_ayah_id' => $validateData['pekerjaan_ayah_id'] ?? null,
                'nama_ibu' => $validateData['nama_ibu'] ?? null,
                'pekerjaan_ibu_id' => $validateData['pekerjaan_ibu_id'] ?? null,
                'no_hp_ortu' => $validateData['no_hp_ortu'] ?? null,
                'kelas_id' => $validateData['kelas_id'],
                'angkatan' => $validateData['angkatan'],
                'status' => $validateData['status'],
                'tahun_ajaran' => $validateData['tahun_ajaran'],
                'thn_ajaran_awal' => $tahunAwal,
                'thn_ajaran_akhir' => $tahunAkhir,
            ]);
            return redirect()->route('siswa.index')->with('successNotif', 'Data siswa "' . $validateData['nama_siswa'] . '" berhasil ditambahkan');
        } catch (\Throwable $th) {
            Log::error('Error saat menyimpan data siswa: ' . $th->getMessage());
            return back()->with('error', 'Data siswa gagal ditambahkan')->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $siswa = Siswa::with(['kelas', 'pekerjaanAyah', 'pekerjaanIbu'])->findOrFail($id);
        
        // Get nilai data grouped by tahun ajaran and kelas
        $nilaiData = BookHead::where('siswa_id', $id)
            ->where('deleted', '0')
            ->with([
                'details' => function($query) {
                    $query->where('deleted', '0')
                        ->with(['mapelHead', 'mapelDetail.mapelHead']);
                }
            ])
            ->orderBy('thn_ajaran_akhir', 'asc')
            ->orderBy('semester', 'asc')
            ->get()
            ->groupBy(function($item) {
                return $item->thn_ajaran_awal . '/' . $item->thn_ajaran_akhir;
            });
        
        return view('pages.siswa.show', compact('siswa', 'nilaiData'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $siswa = Siswa::findOrFail($id);
        $kelas = Kelas::where('deleted', '0')
            ->orderBy('nama_kelas_nomor', 'asc')
            ->orderBy('nama_kelas_sub', 'asc')
            ->get();
        $pekerjaan = Pekerjaan::where('deleted', '0')->orderBy('nama_pekerjaan', 'asc')->get();
        $tahunAjaran = TahunAjaran::where('deleted', '0')->orderBy('tahun_akhir', 'desc')->get();

        return view('pages.siswa.edit', compact('siswa', 'kelas', 'pekerjaan', 'tahunAjaran'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $validateData = $request->validate([
            'foto' => 'nullable|image|mimes:jpg,jpeg,png|max:2048',
            'nis' => 'required|max:50|unique:siswas,nis,' . $id,
            'nisn' => 'required|max:50|unique:siswas,nisn,' . $id,
            'no_kip' => 'nullable|max:50|unique:siswas,no_kip,' . $id,
            'no_kk' => 'nullable|max:50',
            'nama_siswa' => 'required|max:50',
            'jenis_kelamin' => 'required|in:Laki-Laki,Perempuan',
            'tempat_lahir' => 'nullable|max:50',
            'tanggal_lahir' => 'nullable|date',
            'agama' => 'required|in:Islam,Kristen,Hindu,Buddha,Konghucu',
            'alamat' => 'nullable',
            'no_hp_siswa' => 'nullable|max:15',
            'nama_ayah' => 'nullable|max:50',
            'pekerjaan_ayah_id' => 'nullable|exists:pekerjaans,id',
            'nama_ibu' => 'nullable|max:50',
            'pekerjaan_ibu_id' => 'nullable|exists:pekerjaans,id',
            'no_hp_ortu' => 'nullable|max:15',
            'kelas_id' => 'required|exists:kelas,id',
            'angkatan' => 'required',
            'status' => 'required|in:Aktif,Non-Aktif',
            'tahun_ajaran' => 'required',
        ], [
            'nis.required' => 'NIS harus diisi',
            'nis.unique' => 'NIS sudah terdaftar',
            'nisn.required' => 'NISN harus diisi',
            'nisn.unique' => 'NISN sudah terdaftar',
            'no_kip.unique' => 'No KIP sudah terdaftar',
            'no_kip.max' => 'No KIP maksimal 50 karakter',
            'no_kk.max' => 'No KK maksimal 50 karakter',
            'nama_siswa.required' => 'Nama siswa harus diisi',
            'jenis_kelamin.required' => 'Jenis kelamin harus dipilih',
            'agama.required' => 'Agama harus dipilih',
            'kelas_id.required' => 'Kelas harus dipilih',
            'angkatan.required' => 'Angkatan harus diisi',
            'status.required' => 'Status harus dipilih',
            'tahun_ajaran.required' => 'Tahun ajaran harus dipilih',
        ]);

        try {
            // Pisahkan tahun ajaran (misal "2025/2026") menjadi awal & akhir
            [$tahunAwal, $tahunAkhir] = explode('/', $validateData['tahun_ajaran']);

            // Update data siswa
            $siswa = Siswa::findOrFail($id);

            // Handle file upload
            if ($request->hasFile('foto')) {
                // Delete old photo if exists
                if ($siswa->foto) {
                    $old_foto_path = public_path('foto_siswa/' . $siswa->foto);
                    if (file_exists($old_foto_path)) {
                        unlink($old_foto_path);
                    }
                }

                $file = $request->file('foto');
                $nama_siswa_cleaned = str_replace([' ', '/'], '_', strtolower($validateData['nama_siswa']));
                $filename = $validateData['nisn'] . '_' . $nama_siswa_cleaned . '.' . $file->getClientOriginalExtension();
                $file->move(public_path('foto_siswa'), $filename);
                $validateData['foto'] = $filename;
            }

            $siswa->update([
                'foto' => $request->hasFile('foto') ? $validateData['foto'] : $siswa->foto,
                'nis' => $validateData['nis'],
                'nisn' => $validateData['nisn'],
                'no_kip' => $validateData['no_kip'] ?? null,
                'no_kk' => $validateData['no_kk'] ?? null,
                'nama_siswa' => $validateData['nama_siswa'],
                'jenis_kelamin' => $validateData['jenis_kelamin'],
                'tempat_lahir' => $validateData['tempat_lahir'] ?? null,
                'tanggal_lahir' => $validateData['tanggal_lahir'] ?? null,
                'agama' => $validateData['agama'],
                'alamat' => $validateData['alamat'] ?? null,
                'no_hp_siswa' => $validateData['no_hp_siswa'] ?? null,
                'nama_ayah' => $validateData['nama_ayah'] ?? null,
                'pekerjaan_ayah_id' => $validateData['pekerjaan_ayah_id'] ?? null,
                'nama_ibu' => $validateData['nama_ibu'] ?? null,
                'pekerjaan_ibu_id' => $validateData['pekerjaan_ibu_id'] ?? null,
                'no_hp_ortu' => $validateData['no_hp_ortu'] ?? null,
                'kelas_id' => $validateData['kelas_id'],
                'angkatan' => $validateData['angkatan'],
                'status' => $validateData['status'],
                'tahun_ajaran' => $validateData['tahun_ajaran'],
                'thn_ajaran_awal' => $tahunAwal,
                'thn_ajaran_akhir' => $tahunAkhir,
            ]);

            return redirect()->route('siswa.index')->with('successNotif', 'Data siswa "' . $validateData['nama_siswa'] . '" berhasil diperbarui');
        } catch (\Throwable $th) {
            Log::error('Error saat memperbarui data siswa: ' . $th->getMessage());
            return back()->with('error', 'Data siswa gagal diperbarui')->withInput();
        }
    }


    /**
     * Check if siswa is being used
     */
    public function checkUsage(string $id)
    {
        try {
            $siswa = Siswa::findOrFail($id);
            
            // Check if used in book_heads table
            $usedInBookHeads = BookHead::where('siswa_id', $siswa->id)
                ->where('deleted', '0')
                ->exists();
            
            return response()->json([
                'used' => $usedInBookHeads,
                'nama_siswa' => $siswa->nama_siswa
            ]);
        } catch (\Throwable $th) {
            return response()->json([
                'error' => 'Terjadi kesalahan saat memeriksa data'
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        try {
            $siswa = Siswa::findOrFail($id);
            
            // Check if used in book_heads
            $usedInBookHeads = BookHead::where('siswa_id', $siswa->id)
                ->where('deleted', '0')
                ->exists();
            
            if ($usedInBookHeads) {
                return back()->with('error', 'Data siswa "' . $siswa->nama_siswa . '" tidak dapat dihapus karena sudah digunakan pada buku raport');
            }
            
            // Soft delete
            $siswa->update(['deleted' => '1']);
            
            return redirect()->route('siswa.index')->with('successNotif', 'Data siswa "' . $siswa->nama_siswa . '" berhasil dihapus');
        } catch (\Throwable $th) {
            Log::error('Error saat menghapus data siswa: ' . $th->getMessage());
            return back()->with('error', 'Data siswa gagal dihapus');
        }
    }
}
