<?php

namespace App\Http\Controllers;

use App\Models\Siswa;
use App\Models\Kelas;
use App\Models\BookHead;
use App\Models\BookDetail;
use App\Models\MapelHead;
use App\Models\MapelKelas;
use App\Models\TahunAjaran;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class RekapNilaiController extends Controller
{
    public function index(Request $request)
    {
        $user = Auth::user();

        // Get selected NISN from request
        $selectedNisn = $request->input('nisn');

        // Get all active students for the select dropdown
        $allStudents = Siswa::where('deleted', '0')
            ->where('status', 'Aktif')
            ->orderBy('nama_siswa', 'asc')
            ->get();

        // If no NISN selected, return with empty data
        if (!$selectedNisn) {
            return view('pages.rekap_nilai.index', [
                'allStudents' => $allStudents,
                'selectedNisn' => null,
                'siswaData' => null,
                'rekapData' => collect()
            ]);
        }

        // Get selected student data
        $siswaData = Siswa::with('kelas')
            ->where('nisn', $selectedNisn)
            ->where('deleted', '0')
            ->first();

        if (!$siswaData) {
            return view('pages.rekap_nilai.index', [
                'allStudents' => $allStudents,
                'selectedNisn' => $selectedNisn,
                'siswaData' => null,
                'rekapData' => collect()
            ])->with('error', 'Siswa tidak ditemukan.');
        }

        // Get all BookHead data for this student (from class 7 to 9, both semesters)
        $bookHeads = BookHead::with(['details'])
            ->where('siswa_id', $siswaData->id)
            ->where('deleted', '0')
            ->orderBy('thn_ajaran_awal', 'asc')
            // ->orderBy('kelas_id', 'asc')
            ->get();

        // Group data by kelas and semester
        $rekapData = [];

        foreach ($bookHeads as $bookHead) {
            $kelas = $bookHead->kelas;
            if (!$kelas) continue;

            $kelasNomor = $kelas->nama_kelas_nomor;
            $semester = $bookHead->semester;

            // Format kelas display
            $roman = trim($kelas->nama_kelas_romawi ?? '');
            $sub = trim($kelas->nama_kelas_sub ?? '');
            $number = trim((string) ($kelas->nama_kelas_nomor ?? ''));

            $romanWithSub = $roman;
            if ($sub !== '') {
                $romanWithSub = trim($romanWithSub . ' ' . $sub);
            }

            $numberWithSub = $number;
            if ($sub !== '') {
                $numberWithSub .= $sub;
            }

            $kelasDisplay = $romanWithSub;
            if ($romanWithSub !== '' && $numberWithSub !== '') {
                $kelasDisplay .= ' (' . $numberWithSub . ')';
            }

            // Get mapel for this kelas
            $mapelAssignments = MapelKelas::where('kelas_id', $kelas->id)->get();

            // Separate mapel heads and details
            $mapelHeadIds = $mapelAssignments->where('mapel_type', 'head')->pluck('mapel_id')->toArray();
            $mapelDetailIds = $mapelAssignments->where('mapel_type', 'detail')->pluck('mapel_id')->toArray();

            // Get MapelHead with their details
            $mapelHeads = MapelHead::with(['details' => function ($query) use ($mapelDetailIds) {
                $query->where('status', 'Aktif')
                    ->where('deleted', '0')
                    ->whereIn('id', $mapelDetailIds);
            }])
                ->where('status', 'Aktif')
                ->where('deleted', '0')
                ->whereIn('id', $mapelHeadIds)
                ->orderBy('urutan', 'asc')
                ->get();

            // Build bookDetails map
            $bookDetails = $bookHead->details->keyBy(function ($detail) {
                return $detail->mapel_detail_id ?? 'head_' . $detail->mapel_head_id;
            });

            // Organize data
            $key = $kelasNomor . '_' . $semester;

            if (!isset($rekapData[$key])) {
                $rekapData[$key] = [
                    'kelas_display' => $kelasDisplay,
                    'kelas_nomor' => $kelasNomor,
                    'semester' => $semester,
                    'tahun_ajaran' => $bookHead->thn_ajaran_awal . '/' . $bookHead->thn_ajaran_akhir,
                    'mapel_heads' => $mapelHeads,
                    'book_head' => $bookHead,
                    'book_details' => $bookDetails
                ];
            }
        }

        // Sort by kelas and semester
        $rekapData = collect($rekapData)->sortBy([
            ['kelas_nomor', 'asc'],
            ['semester', 'asc']
        ])->values();

        return view('pages.rekap_nilai.index', [
            'allStudents' => $allStudents,
            'selectedNisn' => $selectedNisn,
            'siswaData' => $siswaData,
            'rekapData' => $rekapData
        ]);
    }
}
