<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;
use Carbon\Carbon;
use Illuminate\Support\Str;
use App\Models\Siswa;
use App\Models\PeriodeWaliKelas;
use App\Models\WaliKelas;
use App\Models\Ekskuls;
use App\Models\KetNilai;
use App\Models\KepalaSekolah;
use App\Models\RaportHead;
use App\Models\RaportDetail;
use App\Models\RaportEkskul;
use App\Models\RaportPrestasi;
use App\Models\MapelHead;
use App\Models\MapelKelas;
use App\Models\TahunAjaran;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Auth;

class RaportController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        // Get active tahun ajaran from TahunAjaran model
        $tahunAjaranAktif = TahunAjaran::where('status', 'Aktif')
            ->where('deleted', '0')
            ->first();

        // Set default tahun ajaran
        if ($tahunAjaranAktif) {
            $defaultTahunAjaran = $tahunAjaranAktif->tahun_awal . '/' . $tahunAjaranAktif->tahun_akhir;
        } else {
            $tahunSekarang = date('Y');
            $defaultTahunAjaran = $tahunSekarang . '/' . ($tahunSekarang + 1);
        }

        $selectedTahunAjaran = $request->get('tahun_ajaran', $defaultTahunAjaran);

        // Get filter parameters
        $selectedKelas = $request->get('kelasSelect', 'all');
        $selectedSemester = $request->get('semesterSelect', 'all');

        // Parse tahun ajaran
        $tahunParts = explode('/', $selectedTahunAjaran);
        $tahunSekarang = date('Y');
        $thnAjaranAwal = isset($tahunParts[0]) ? (int) $tahunParts[0] : $tahunSekarang;
        $thnAjaranAkhir = isset($tahunParts[1]) ? (int) $tahunParts[1] : $tahunSekarang + 1;

        // Get students grouped by kelas for the selected tahun ajaran
        $kelasData = [];

        // Always load data for the selected tahun ajaran
        $isAlumniMode = false;

        if ($selectedTahunAjaran) {
            // Ambil siswa untuk tahun ajaran yang dipilih
            $studentsQuery = Siswa::with(['kelas' => function ($query) {
                $query->orderBy('nama_kelas_nomor', 'asc')
                    ->orderBy('nama_kelas_sub', 'asc');
            }])
                ->orderBy('nama_siswa', 'asc')
                ->where('thn_ajaran_awal', $thnAjaranAwal)
                ->where('thn_ajaran_akhir', $thnAjaranAkhir)
                ->where('deleted', '0')
                ->where('status', '!=', 'Non-Aktif');

            // Filter berdasarkan role user
            $user = Auth::user();
            if ($user->role === 'Wali') {
                // Jika user adalah Wali, hanya tampilkan siswa dari kelas yang diampu
                $waliKelas = WaliKelas::where('user_id', $user->id)
                    ->where('deleted', '0')
                    ->first();

                if ($waliKelas) {
                    $kelasIds = PeriodeWaliKelas::where('wali_kelas_id', $waliKelas->id)
                        ->where('thn_ajaran_awal', $thnAjaranAwal)
                        ->where('thn_ajaran_akhir', $thnAjaranAkhir)
                        ->where('deleted', '0')
                        ->pluck('kelas_id');

                    $studentsQuery->whereIn('kelas_id', $kelasIds);
                } else {
                    // Jika tidak ada data wali kelas, return empty
                    $studentsQuery->whereRaw('1 = 0');
                }
            }

            // Apply kelas filter if not 'all'
            if ($selectedKelas !== 'all') {
                $studentsQuery->whereHas('kelas', function ($query) use ($selectedKelas) {
                    $query->where('nama_kelas_nomor', $selectedKelas);
                });
            }

            $studentsCollection = $studentsQuery->get();

            if ($studentsCollection->isNotEmpty()) {
                $studentIds = $studentsCollection->pluck('id');

                $raportHeads = RaportHead::select('id', 'siswa_id', 'semester')
                    ->whereIn('siswa_id', $studentIds)
                    ->where('thn_ajaran_awal', $thnAjaranAwal)
                    ->where('thn_ajaran_akhir', $thnAjaranAkhir)
                    ->where('deleted', '0')
                    ->get();

                $statusMap = [];
                foreach ($raportHeads as $head) {
                    $statusMap[$head->siswa_id][$head->semester] = $head->id;
                }

                $studentsCollection = $studentsCollection->map(function ($student) use ($statusMap) {
                    $raportHeadIds = [
                        'Ganjil' => $statusMap[$student->id]['Ganjil'] ?? null,
                        'Genap' => $statusMap[$student->id]['Genap'] ?? null,
                    ];

                    $student->setAttribute('raport_status', [
                        'Ganjil' => !is_null($raportHeadIds['Ganjil']),
                        'Genap' => !is_null($raportHeadIds['Genap']),
                    ]);

                    $student->setAttribute('raport_head_ids', $raportHeadIds);

                    return $student;
                });
            }

            $isAlumniMode = $studentsCollection->isNotEmpty() &&
                $studentsCollection->every(fn($siswa) => $siswa->status === 'Alumni');

            if ($isAlumniMode) {
                $groupedByAngkatan = $studentsCollection
                    ->sortByDesc('angkatan')
                    ->groupBy('angkatan')
                    ->sortKeysDesc();

                foreach ($groupedByAngkatan as $angkatan => $students) {
                    $kelasData[] = [
                        'kelas' => null,
                        'kelas_name' => null,
                        'students' => $students->sortBy('nama_siswa')->values(),
                        'angkatan' => $angkatan,
                        'wali_kelas' => '-',
                        'group_label' => 'Angkatan',
                        'group_value' => $angkatan,
                        'group_key' => 'angkatan-' . $angkatan,
                    ];
                }
            } else {
                $kelasWithStudents = $studentsCollection->groupBy('kelas_id');

                $sortedKelasData = $kelasWithStudents->sortBy(function ($students) {
                    $kelas = $students->first()->kelas;
                    return $kelas->nama_kelas_nomor . ($kelas->nama_kelas_sub ?? '');
                });

                foreach ($sortedKelasData as $kelasId => $students) {
                    $kelas = $students->first()->kelas;

                    // Get wali kelas untuk kelas dan tahun ajaran ini
                    $periodeWaliKelas = PeriodeWaliKelas::with(['waliKelas.user'])
                        ->where('kelas_id', $kelasId)
                        ->where('thn_ajaran_awal', $thnAjaranAwal)
                        ->where('thn_ajaran_akhir', $thnAjaranAkhir)
                        ->where('deleted', '0')
                        ->first();

                    $roman = trim($kelas->nama_kelas_romawi ?? '');
                    $sub = trim($kelas->nama_kelas_sub ?? '');
                    $number = trim((string) ($kelas->nama_kelas_nomor ?? ''));

                    $romanWithSub = $roman;
                    if ($sub !== '') {
                        $romanWithSub = trim($romanWithSub . ' ' . $sub);
                    }

                    $numberWithSub = $number;
                    if ($sub !== '') {
                        $numberWithSub .= $sub;
                    }

                    $kelasDisplay = $romanWithSub;
                    if ($romanWithSub !== '' && $numberWithSub !== '') {
                        $kelasDisplay .= ' (' . $numberWithSub . ')';
                    }

                    $kelasData[] = [
                        'kelas' => $kelas,
                        'kelas_name' => $kelas?->nama_kelas,
                        'students' => $students->values(),
                        'angkatan' => $students->first()->angkatan,
                        'wali_kelas' => $periodeWaliKelas ? $periodeWaliKelas->waliKelas->user->name : 'Belum Ditentukan',
                        'group_label' => 'Kelas',
                        'group_value' => $kelasDisplay ?: $kelas?->nama_kelas,
                        'group_key' => $kelas ? 'kelas-' . $kelas->id : 'kelas-unknown',
                    ];
                }
            }
        }
        if ($request->ajax()) {
            return response()->json([
                'kelasData' => $kelasData,
                'selectedTahunAjaran' => $selectedTahunAjaran,
                'selectedKelas' => $selectedKelas,
                'selectedSemester' => $selectedSemester,
                'isAlumniMode' => $isAlumniMode,
            ]);
        }

        // Get all TahunAjaran data
        $dataTahunAjaran = TahunAjaran::where('deleted', '0')
            ->orderBy('tahun_akhir', 'desc')
            ->get();

        // Get progress wali kelas (hanya untuk wali yang login)
        $waliKelasProgress = null;
        if ($user->role === 'Wali') {
            $waliKelas = WaliKelas::where('user_id', $user->id)
                ->where('deleted', '0')
                ->first();

            if ($waliKelas) {
                $periodeWaliKelas = PeriodeWaliKelas::with(['kelas'])
                    ->where('wali_kelas_id', $waliKelas->id)
                    ->where('thn_ajaran_awal', $thnAjaranAwal)
                    ->where('thn_ajaran_akhir', $thnAjaranAkhir)
                    ->where('deleted', '0')
                    ->first();

                if ($periodeWaliKelas) {
                    $kelas = $periodeWaliKelas->kelas;

                    // Get total students in this class
                    $totalSiswa = Siswa::where('kelas_id', $kelas->id)
                        ->where('thn_ajaran_awal', $thnAjaranAwal)
                        ->where('thn_ajaran_akhir', $thnAjaranAkhir)
                        ->where('status', 'Aktif')
                        ->where('deleted', '0')
                        ->count();

                    // Count raport Ganjil
                    $totalSudahInputGanjil = RaportHead::where('wali_kelas_id', $waliKelas->id)
                        ->where('semester', 'Ganjil')
                        ->where('thn_ajaran_awal', $thnAjaranAwal)
                        ->where('thn_ajaran_akhir', $thnAjaranAkhir)
                        ->where('deleted', '0')
                        ->whereIn('siswa_id', function ($query) use ($kelas, $thnAjaranAwal, $thnAjaranAkhir) {
                            $query->select('id')
                                ->from('siswas')
                                ->where('kelas_id', $kelas->id)
                                ->where('thn_ajaran_awal', $thnAjaranAwal)
                                ->where('thn_ajaran_akhir', $thnAjaranAkhir)
                                ->where('status', 'Aktif')
                                ->where('deleted', '0');
                        })
                        ->count();

                    // Count raport Genap
                    $totalSudahInputGenap = RaportHead::where('wali_kelas_id', $waliKelas->id)
                        ->where('semester', 'Genap')
                        ->where('thn_ajaran_awal', $thnAjaranAwal)
                        ->where('thn_ajaran_akhir', $thnAjaranAkhir)
                        ->where('deleted', '0')
                        ->whereIn('siswa_id', function ($query) use ($kelas, $thnAjaranAwal, $thnAjaranAkhir) {
                            $query->select('id')
                                ->from('siswas')
                                ->where('kelas_id', $kelas->id)
                                ->where('thn_ajaran_awal', $thnAjaranAwal)
                                ->where('thn_ajaran_akhir', $thnAjaranAkhir)
                                ->where('status', 'Aktif')
                                ->where('deleted', '0');
                        })
                        ->count();

                    // Format kelas display
                    $roman = trim($kelas->nama_kelas_romawi ?? '');
                    $sub = trim($kelas->nama_kelas_sub ?? '');
                    $number = trim((string) ($kelas->nama_kelas_nomor ?? ''));

                    $romanWithSub = $roman;
                    if ($sub !== '') {
                        $romanWithSub = trim($romanWithSub . ' ' . $sub);
                    }

                    $numberWithSub = $number;
                    if ($sub !== '') {
                        $numberWithSub .= $sub;
                    }

                    $kelasDisplay = $romanWithSub;
                    if ($romanWithSub !== '' && $numberWithSub !== '') {
                        $kelasDisplay .= ' (' . $numberWithSub . ')';
                    }

                    $waliKelasProgress = [
                        'wali_kelas_nama' => $user->name,
                        'kelas_display' => $kelasDisplay ?: $kelas->nama_kelas,
                        'total_siswa' => $totalSiswa,
                        'ganjil' => [
                            'sudah_input' => $totalSudahInputGanjil,
                            'harus_input' => $totalSiswa,
                            'progress_percentage' => $totalSiswa > 0 ? round(($totalSudahInputGanjil / $totalSiswa) * 100, 1) : 0,
                        ],
                        'genap' => [
                            'sudah_input' => $totalSudahInputGenap,
                            'harus_input' => $totalSiswa,
                            'progress_percentage' => $totalSiswa > 0 ? round(($totalSudahInputGenap / $totalSiswa) * 100, 1) : 0,
                        ],
                    ];
                }
            }
        }

        return view('pages.raport.index', compact('kelasData', 'selectedTahunAjaran', 'selectedKelas', 'selectedSemester', 'isAlumniMode', 'dataTahunAjaran', 'waliKelasProgress'));
    }

    public function create(Request $request, string $id)
    {
        $siswa = Siswa::with('kelas')->findOrFail($id);

        $semester = $request->query('semester', 'Ganjil');
        if (!in_array($semester, ['Ganjil', 'Genap'])) {
            $semester = 'Ganjil';
        }

        $kelasId = $siswa->kelas_id;

        // Get mapel heads that are assigned to this kelas
        $mapelHeadIds = \App\Models\MapelKelas::where('kelas_id', $kelasId)
            ->where('mapel_type', 'head')
            ->pluck('mapel_id');

        // Get mapel detail ids that are assigned to this kelas
        $mapelDetailIds = \App\Models\MapelKelas::where('kelas_id', $kelasId)
            ->where('mapel_type', 'detail')
            ->pluck('mapel_id');

        $mapelHeads = \App\Models\MapelHead::with(['details' => function ($query) use ($mapelDetailIds) {
            $query->where('status', 'Aktif')
                ->where('deleted', '0')
                ->whereIn('id', $mapelDetailIds)
                ->orderBy('nama_mapel_detail');
        }])
            ->where('status', 'Aktif')
            ->where('deleted', '0')
            ->whereIn('id', $mapelHeadIds)
            ->orderBy('nama_mapel')
            ->get();

        $ekskuls = Ekskuls::where('status', 'Aktif')
            ->where('deleted', '0')
            ->orderBy('nama_ekskul')
            ->get();

        $ketNilaiList = KetNilai::where('deleted', '0')
            ->orderBy('nama_keterangan')
            ->get();

        // $promotionOptions = $this->buildPromotionOptions($siswa->kelas);
        // $promotionOptions = $siswa->kelas->nama_kelas_nomor == 9
        //     ? ['Lulus', 'Tidak Lulus']
        //     : ['Naik Kelas', 'Tidak Naik Kelas'];
        $promotionOptions = $siswa->kelas->nama_kelas_nomor == 9
            ? ['Lulus', 'Tidak Lulus']
            : $this->buildPromotionOptions($siswa->kelas);

        // Get wali kelas untuk siswa ini
        $kelas = $siswa->kelas;
        $periodeWali = PeriodeWaliKelas::with('waliKelas')
            ->where('kelas_id', $kelas?->id)
            ->where('thn_ajaran_awal', $siswa->thn_ajaran_awal)
            ->where('thn_ajaran_akhir', $siswa->thn_ajaran_akhir)
            ->where('deleted', '0')
            ->first();

        $waliKelas = $periodeWali?->waliKelas;

        // Get kepala sekolah aktif
        $kepalaSekolah = KepalaSekolah::where('status', 'Aktif')
            ->where('deleted', '0')
            ->orderByDesc('id')
            ->first();

        return view('pages.raport.create', [
            'siswa' => $siswa,
            'mapelHeads' => $mapelHeads,
            'ekskuls' => $ekskuls,
            'ketNilaiList' => $ketNilaiList,
            'promotionOptions' => $promotionOptions,
            'selectedSemester' => $semester,
            'waliKelas' => $waliKelas,
            'kepalaSekolah' => $kepalaSekolah,
        ]);
    }

    protected function buildPromotionOptions($kelas)
    {
        $options = [];

        if ($kelas) {
            $romanLookup = [
                1 => 'I (1)',
                2 => 'II (2)',
                3 => 'III (3)',
                4 => 'IV (4)',
                5 => 'V (5)',
                6 => 'VI (6)',
                7 => 'VII (7)',
                8 => 'VIII (8)',
                9 => 'IX (9)',
                10 => 'X (10)',
                11 => 'XI (11)',
                12 => 'XII (12)',
            ];

            $currentNumber = (int) ($kelas->nama_kelas_nomor ?? 0);
            $nextNumber = $currentNumber ? $currentNumber + 1 : null;
            $nextRoman = $nextNumber ? ($romanLookup[$nextNumber] ?? null) : null;

            $promotionLabel = 'Naik Ke Kelas';
            if ($nextRoman) {
                $promotionLabel .= ' ' . $nextRoman;
            }

            $options[] = trim($promotionLabel);
        }

        $options[] = 'Tidak Naik Kelas';

        return $options;
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'siswa_id' => ['required', 'exists:siswas,id'],
            'total_nilai_akhir' => ['nullable', 'numeric'],
            'ketidakhadiran.sakit' => ['nullable', 'numeric', 'min:0'],
            'ketidakhadiran.izin' => ['nullable', 'numeric', 'min:0'],
            'ketidakhadiran.alpa' => ['nullable', 'numeric', 'min:0'],
            'catatan_pendidik' => ['nullable', 'string'],
            'keterangan_kenaikan_kelas' => ['nullable', 'string', 'max:100'],
            'tanggal_titimangsa' => ['required', 'date'],
            'semester' => ['nullable', 'in:Ganjil,Genap'],
            'mapel_heads' => ['nullable', 'array'],
            'mapel_heads.*.nilai_akhir' => ['nullable', 'numeric', 'between:0,100'],
            'mapel_heads.*.capaian_kompetensi' => ['nullable', 'string'],
            'mapel_details' => ['nullable', 'array'],
            'mapel_details.*.nilai_akhir' => ['nullable', 'numeric', 'between:0,100'],
            'mapel_details.*.capaian_kompetensi' => ['nullable', 'string'],
            'ekstrakurikuler' => ['nullable', 'array'],
            'ekstrakurikuler.*.kegiatan_id' => ['nullable', 'exists:ekskuls,id'],
            'ekstrakurikuler.*.nilai_id' => ['nullable', 'exists:ket_nilais,id'],
            'ekstrakurikuler.*.keterangan' => ['nullable', 'string'],
            'prestasi' => ['nullable', 'array'],
            'prestasi.*.jenis' => ['nullable', 'string', 'max:250'],
            'prestasi.*.keterangan' => ['nullable', 'string'],
        ]);

        $siswa = Siswa::with('kelas')->findOrFail($validated['siswa_id']);
        $semesterInput = $validated['semester'] ?? $request->input('semester', 'Ganjil');

        DB::transaction(function () use ($validated, $siswa, $semesterInput) {
            $kelas = $siswa->kelas;
            $periodeWali = PeriodeWaliKelas::with('waliKelas')
                ->where('kelas_id', $kelas?->id)
                ->where('thn_ajaran_awal', $siswa->thn_ajaran_awal)
                ->where('thn_ajaran_akhir', $siswa->thn_ajaran_akhir)
                ->where('deleted', '0')
                ->first();

            $waliKelasId = $periodeWali?->wali_kelas_id;
            if (!$waliKelasId) {
                throw ValidationException::withMessages([
                    'wali_kelas_id' => 'Data wali kelas untuk siswa ini belum ditetapkan.',
                ]);
            }

            $kepalaSekolahId = KepalaSekolah::where('status', 'Aktif')
                ->where('deleted', '0')
                ->orderByDesc('id')
                ->value('id');

            if (!$kepalaSekolahId) {
                throw ValidationException::withMessages([
                    'kepala_sekolah_id' => 'Data kepala madrasah belum tersedia.',
                ]);
            }

            $fase = $this->determineFase($kelas?->nama_kelas_nomor);
            $semester = in_array($semesterInput, ['Ganjil', 'Genap']) ? $semesterInput : 'Ganjil';

            $raportHead = RaportHead::create([
                'siswa_id' => $siswa->id,
                'thn_ajaran_awal' => $siswa->thn_ajaran_awal,
                'thn_ajaran_akhir' => $siswa->thn_ajaran_akhir,
                'kepala_sekolah_id' => $kepalaSekolahId,
                'wali_kelas_id' => $waliKelasId,
                'fase' => $fase,
                'semester' => $semester,
                'jml_sakit' => data_get($validated, 'ketidakhadiran.sakit', 0),
                'jml_izin' => data_get($validated, 'ketidakhadiran.izin', 0),
                'jml_alpa' => data_get($validated, 'ketidakhadiran.alpa', 0),
                'catatan_pendidik' => $validated['catatan_pendidik'] ?? null,
                'keterangan_kenaikan' => $validated['keterangan_kenaikan_kelas'] ?? null,
                'tanggal_titimangsa' => $validated['tanggal_titimangsa'],
                'jml_nilai_akhir' => $validated['total_nilai_akhir'] ?? 0,
            ]);

            $promotionNote = $validated['keterangan_kenaikan_kelas'] ?? null;
            if (!is_null($promotionNote)) {
                if (Str::startsWith($promotionNote, 'Tidak')) {
                    $siswa->update(['keterangan_kenaikan' => $promotionNote]);
                } else {
                    $siswa->update(['keterangan_kenaikan' => null]);
                }
            }

            foreach ($validated['mapel_heads'] ?? [] as $mapelHeadId => $data) {
                if (empty($data['nilai_akhir']) && empty($data['capaian_kompetensi'])) {
                    continue;
                }

                RaportDetail::create([
                    'raport_head_id' => $raportHead->id,
                    'mapel_head_id' => $mapelHeadId,
                    'mapel_detail_id' => null,
                    'nilai_akhir' => $data['nilai_akhir'] ?? null,
                    'deskripsi' => $data['capaian_kompetensi'] ?? null,
                ]);
            }

            foreach ($validated['mapel_details'] ?? [] as $mapelDetailId => $data) {
                if (empty($data['nilai_akhir']) && empty($data['capaian_kompetensi'])) {
                    continue;
                }

                RaportDetail::create([
                    'raport_head_id' => $raportHead->id,
                    'mapel_head_id' => null,
                    'mapel_detail_id' => $mapelDetailId,
                    'nilai_akhir' => $data['nilai_akhir'] ?? null,
                    'deskripsi' => $data['capaian_kompetensi'] ?? null,
                ]);
            }

            foreach ($validated['ekstrakurikuler'] ?? [] as $row) {
                if (empty($row['kegiatan_id']) && empty($row['nilai_id']) && empty($row['keterangan'])) {
                    continue;
                }

                if (empty($row['kegiatan_id']) || empty($row['nilai_id'])) {
                    throw ValidationException::withMessages([
                        'ekstrakurikuler' => 'Setiap ekstrakurikuler harus memiliki kegiatan dan nilai.',
                    ]);
                }

                RaportEkskul::create([
                    'raport_head_id' => $raportHead->id,
                    'ekskul_id' => $row['kegiatan_id'],
                    'ket_nilai' => $row['nilai_id'] ?? null,
                    'keterangan' => $row['keterangan'] ?? null,
                ]);
            }

            foreach ($validated['prestasi'] ?? [] as $row) {
                if (empty($row['jenis']) && empty($row['keterangan'])) {
                    continue;
                }

                RaportPrestasi::create([
                    'raport_head_id' => $raportHead->id,
                    'jenis_prestasi' => $row['jenis'] ?? null,
                    'keterangan' => $row['keterangan'] ?? null,
                ]);
            }
        });

        return redirect()->route('raport.index')->with('successNotif', 'Raport berhasil disimpan.');
    }

    protected function determineFase($kelasNomor): string
    {
        $faseMap = [
            7 => 'D',
            8 => 'E',
            9 => 'F',
            10 => 'G',
            11 => 'H',
            12 => 'I',
        ];

        return $faseMap[$kelasNomor ?? null] ?? '-';
    }

    /**
     * Display the specified resource.
     */
    public function show(Request $request, string $id)
    {
        Carbon::setLocale('id');

        $raportHead = RaportHead::with([
            'siswa.kelas',
            'details.mapelHead',
            'details.mapelDetail.mapelHead',
            'ekskuls.ekskul',
            'ekskuls.ketNilai',
            'prestasis',
            'kepalaSekolah',
            'waliKelas',
        ])
            ->where('deleted', '0')
            ->find($id);

        if ($raportHead) {
            $siswa = $raportHead->siswa;
            if (!$siswa) {
                abort(404, 'Data siswa untuk raport ini tidak ditemukan.');
            }
        } else {
            $siswa = Siswa::with('kelas')->findOrFail($id);

            $semesterFilter = $request->get('semester');

            $raportHead = RaportHead::with([
                'siswa.kelas',
                'details.mapelHead',
                'details.mapelDetail.mapelHead',
                'ekskuls.ekskul',
                'ekskuls.ketNilai',
                'prestasis',
                'kepalaSekolah',
                'waliKelas',
            ])
                ->where('siswa_id', $siswa->id)
                ->where('deleted', '0');

            if ($semesterFilter && in_array($semesterFilter, ['Ganjil', 'Genap'])) {
                $raportHead->where('semester', $semesterFilter);
            }

            $raportHead = $raportHead->latest('tanggal_titimangsa')->first();
        }

        $kelasId = $siswa->kelas_id;

        // Get mapel heads that are assigned to this kelas
        $mapelHeadIds = \App\Models\MapelKelas::where('kelas_id', $kelasId)
            ->where('mapel_type', 'head')
            ->pluck('mapel_id');

        // Get mapel detail ids that are assigned to this kelas
        $mapelDetailIds = \App\Models\MapelKelas::where('kelas_id', $kelasId)
            ->where('mapel_type', 'detail')
            ->pluck('mapel_id');

        $mapelHeads = \App\Models\MapelHead::with(['details' => function ($query) use ($mapelDetailIds) {
            $query->where('status', 'Aktif')
                ->where('deleted', '0')
                ->whereIn('id', $mapelDetailIds)
                ->orderBy('nama_mapel_detail');
        }])
            ->where('status', 'Aktif')
            ->where('deleted', '0')
            ->whereIn('id', $mapelHeadIds)
            ->orderBy('nama_mapel')
            ->get();

        $headScores = collect();
        $detailScores = collect();
        $ekskulPengisian = collect();
        $prestasiPengisian = collect();
        $ketidakhadiran = [
            'sakit' => 0,
            'izin' => 0,
            'alpa' => 0,
        ];
        $formattedTanggal = null;

        if ($raportHead) {
            $headScores = $raportHead->details
                ->whereNotNull('mapel_head_id')
                ->keyBy('mapel_head_id');

            $detailScores = $raportHead->details
                ->whereNotNull('mapel_detail_id')
                ->keyBy('mapel_detail_id');

            $ekskulPengisian = $raportHead->ekskuls;
            $prestasiPengisian = $raportHead->prestasis;

            $ketidakhadiran = [
                'sakit' => $raportHead->jml_sakit ?? 0,
                'izin' => $raportHead->jml_izin ?? 0,
                'alpa' => $raportHead->jml_alpa ?? 0,
            ];

            $formattedTanggal = $raportHead->tanggal_titimangsa
                ? Carbon::parse($raportHead->tanggal_titimangsa)->translatedFormat('d F Y')
                : null;

            $raportMapelHeadIds = $raportHead->details
                ->pluck('mapel_head_id')
                ->filter();

            $raportMapelHeadIds = $raportMapelHeadIds->merge(
                $raportHead->details
                    ->pluck('mapelDetail.mapelHead.id')
                    ->filter()
            )->unique();

            $missingHeadIds = $raportMapelHeadIds->diff($mapelHeads->pluck('id'));

            if ($missingHeadIds->isNotEmpty()) {
                $additionalHeads = \App\Models\MapelHead::with(['details' => function ($query) {
                    $query->orderBy('nama_mapel_detail');
                }])
                    ->whereIn('id', $missingHeadIds)
                    ->get();

                $mapelHeads = $mapelHeads
                    ->merge($additionalHeads)
                    ->unique('id')
                    ->sortBy('nama_mapel')
                    ->values();
            }
        }

        $kelas = $siswa->kelas;
        $roman = trim($kelas->nama_kelas_romawi ?? '');
        $sub = trim($kelas->nama_kelas_sub ?? '');
        $number = trim((string) ($kelas->nama_kelas_nomor ?? ''));

        $kelasDisplay = $roman;
        if ($sub !== '') {
            $kelasDisplay = trim($kelasDisplay . ' ' . $sub);
        }

        $numberWithSub = $number;
        if ($sub !== '') {
            $numberWithSub .= $sub;
        }

        if ($kelasDisplay && $numberWithSub) {
            $kelasDisplay .= ' (' . $numberWithSub . ')';
        }

        return view('pages.raport.detail', [
            'siswa' => $siswa,
            'raportHead' => $raportHead,
            'mapelHeads' => $mapelHeads,
            'headScores' => $headScores,
            'detailScores' => $detailScores,
            'ekskulPengisian' => $ekskulPengisian,
            'prestasiPengisian' => $prestasiPengisian,
            'ketidakhadiran' => $ketidakhadiran,
            'formattedTanggal' => $formattedTanggal,
            'kelasDisplay' => $kelasDisplay ?: '-',
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        Carbon::setLocale('id');

        $raportHead = RaportHead::with([
            'siswa.kelas',
            'details.mapelHead',
            'details.mapelDetail.mapelHead',
            'ekskuls.ekskul',
            'ekskuls.ketNilai',
            'prestasis',
            'kepalaSekolah',
            'waliKelas',
        ])
            ->where('deleted', '0')
            ->findOrFail($id);

        $siswa = $raportHead->siswa;

        if (!$siswa) {
            abort(404, 'Data siswa untuk raport ini tidak ditemukan.');
        }

        $kelasId = $siswa->kelas_id;

        // Get mapel heads that are assigned to this kelas
        $mapelHeadIds = \App\Models\MapelKelas::where('kelas_id', $kelasId)
            ->where('mapel_type', 'head')
            ->pluck('mapel_id');

        // Get mapel detail ids that are assigned to this kelas
        $mapelDetailIds = \App\Models\MapelKelas::where('kelas_id', $kelasId)
            ->where('mapel_type', 'detail')
            ->pluck('mapel_id');

        $mapelHeads = MapelHead::with(['details' => function ($query) use ($mapelDetailIds) {
            $query->where('status', 'Aktif')
                ->where('deleted', '0')
                ->whereIn('id', $mapelDetailIds)
                ->orderBy('nama_mapel_detail');
        }])
            ->where('status', 'Aktif')
            ->where('deleted', '0')
            ->whereIn('id', $mapelHeadIds)
            ->orderBy('nama_mapel')
            ->get();

        $headScores = $raportHead->details
            ->whereNotNull('mapel_head_id')
            ->keyBy('mapel_head_id');

        $detailScores = $raportHead->details
            ->whereNotNull('mapel_detail_id')
            ->keyBy('mapel_detail_id');

        $raportMapelHeadIds = $raportHead->details
            ->pluck('mapel_head_id')
            ->filter();

        $raportMapelHeadIds = $raportMapelHeadIds->merge(
            $raportHead->details
                ->pluck('mapelDetail.mapelHead.id')
                ->filter()
        )->unique();

        $missingHeadIds = $raportMapelHeadIds->diff($mapelHeads->pluck('id'));

        if ($missingHeadIds->isNotEmpty()) {
            $additionalHeads = MapelHead::with(['details' => function ($query) {
                $query->orderBy('nama_mapel_detail');
            }])
                ->whereIn('id', $missingHeadIds)
                ->get();

            $mapelHeads = $mapelHeads
                ->merge($additionalHeads)
                ->unique('id')
                ->sortBy('nama_mapel')
                ->values();
        }

        $ekskuls = Ekskuls::where('status', 'Aktif')
            ->where('deleted', '0')
            ->orderBy('nama_ekskul')
            ->get();

        $ketNilaiList = KetNilai::where('deleted', '0')
            ->orderBy('nama_keterangan')
            ->get();

        $kelas = $siswa->kelas;
        $roman = trim($kelas->nama_kelas_romawi ?? '');
        $sub = trim($kelas->nama_kelas_sub ?? '');
        $number = trim((string) ($kelas->nama_kelas_nomor ?? ''));

        $kelasDisplay = $roman;
        if ($sub !== '') {
            $kelasDisplay = trim($kelasDisplay . ' ' . $sub);
        }

        $numberWithSub = $number;
        if ($sub !== '') {
            $numberWithSub .= $sub;
        }

        if ($kelasDisplay && $numberWithSub) {
            $kelasDisplay .= ' (' . $numberWithSub . ')';
        }

        $fase = $this->determineFase($kelas->nama_kelas_nomor ?? null);
        $semester = $raportHead->semester ?: 'Ganjil';
        $tahunAjaran = $siswa->thn_ajaran_awal && $siswa->thn_ajaran_akhir
            ? $siswa->thn_ajaran_awal . '/' . $siswa->thn_ajaran_akhir
            : '-';

        $ketidakhadiran = [
            'sakit' => $raportHead->jml_sakit ?? 0,
            'izin' => $raportHead->jml_izin ?? 0,
            'alpa' => $raportHead->jml_alpa ?? 0,
        ];

        // $promotionOptions = $this->buildPromotionOptions($kelas);
        // $promotionOptions = $siswa->kelas->nama_kelas_nomor == 9
        //     ? ['Lulus', 'Tidak Lulus']
        //     : ['Naik Kelas', 'Tidak Naik Kelas'];
        $promotionOptions = $siswa->kelas->nama_kelas_nomor == 9
            ? ['Lulus', 'Tidak Lulus']
            : $this->buildPromotionOptions($siswa->kelas);

        $ekskulPengisian = $raportHead->ekskuls->values();
        $prestasiPengisian = $raportHead->prestasis->values();

        return view('pages.raport.edit', [
            'siswa' => $siswa,
            'raportHead' => $raportHead,
            'mapelHeads' => $mapelHeads,
            'headScores' => $headScores,
            'detailScores' => $detailScores,
            'ekskulPengisian' => $ekskulPengisian,
            'prestasiPengisian' => $prestasiPengisian,
            'ekskuls' => $ekskuls,
            'ketNilaiList' => $ketNilaiList,
            'kelasDisplay' => $kelasDisplay ?: '-',
            'fase' => $fase,
            'semester' => $semester,
            'tahunAjaran' => $tahunAjaran,
            'ketidakhadiran' => $ketidakhadiran,
            'promotionOptions' => $promotionOptions,
            'selectedPromotion' => $raportHead->keterangan_kenaikan,
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $raportHead = RaportHead::with('siswa.kelas')
            ->where('deleted', '0')
            ->findOrFail($id);

        $validated = $request->validate([
            'siswa_id' => ['required', 'exists:siswas,id'],
            'total_nilai_akhir' => ['nullable', 'numeric'],
            'ketidakhadiran.sakit' => ['nullable', 'numeric', 'min:0'],
            'ketidakhadiran.izin' => ['nullable', 'numeric', 'min:0'],
            'ketidakhadiran.alpa' => ['nullable', 'numeric', 'min:0'],
            'catatan_pendidik' => ['nullable', 'string'],
            'keterangan_kenaikan_kelas' => ['nullable', 'string', 'max:100'],
            'tanggal_titimangsa' => ['required', 'date'],
            'semester' => ['nullable', 'in:Ganjil,Genap'],
            'mapel_heads' => ['nullable', 'array'],
            'mapel_heads.*.nilai_akhir' => ['nullable', 'numeric', 'between:0,100'],
            'mapel_heads.*.capaian_kompetensi' => ['nulla   ble', 'string'],
            'mapel_details' => ['nullable', 'array'],
            'mapel_details.*.nilai_akhir' => ['nullable', 'numeric', 'between:0,100'],
            'mapel_details.*.capaian_kompetensi' => ['nullable', 'string'],
            'ekstrakurikuler' => ['nullable', 'array'],
            'ekstrakurikuler.*.kegiatan_id' => ['nullable', 'exists:ekskuls,id'],
            'ekstrakurikuler.*.nilai_id' => ['nullable', 'exists:ket_nilais,id'],
            'ekstrakurikuler.*.keterangan' => ['nullable', 'string'],
            'prestasi' => ['nullable', 'array'],
            'prestasi.*.jenis' => ['nullable', 'string', 'max:250'],
            'prestasi.*.keterangan' => ['nullable', 'string'],
        ]);

        if ((int) $raportHead->siswa_id !== (int) $validated['siswa_id']) {
            throw ValidationException::withMessages([
                'siswa_id' => 'Data siswa tidak sesuai dengan raport yang diedit.',
            ]);
        }

        $semesterInput = $validated['semester'] ?? $raportHead->semester ?? 'Ganjil';

        $siswa = $raportHead->siswa;

        DB::transaction(function () use ($validated, $raportHead, $siswa, $semesterInput) {
            $raportHead->update([
                'jml_sakit' => data_get($validated, 'ketidakhadiran.sakit', 0),
                'jml_izin' => data_get($validated, 'ketidakhadiran.izin', 0),
                'jml_alpa' => data_get($validated, 'ketidakhadiran.alpa', 0),
                'catatan_pendidik' => $validated['catatan_pendidik'] ?? null,
                'keterangan_kenaikan' => $validated['keterangan_kenaikan_kelas'] ?? null,
                'tanggal_titimangsa' => $validated['tanggal_titimangsa'],
                'semester' => in_array($semesterInput, ['Ganjil', 'Genap']) ? $semesterInput : 'Ganjil',
                'jml_nilai_akhir' => $validated['total_nilai_akhir'] ?? 0,
            ]);

            $promotionNote = $validated['keterangan_kenaikan_kelas'] ?? null;
            if ($siswa && !is_null($promotionNote)) {
                if (Str::startsWith($promotionNote, 'Tidak')) {
                    $siswa->update(['keterangan_kenaikan' => $promotionNote]);
                } elseif (Str::startsWith($promotionNote, 'Naik Ke Kelas') || $promotionNote === 'Lulus') {
                    $siswa->update(['keterangan_kenaikan' => null]);
                }
            }

            $raportHead->details()->update(['deleted' => '1']);

            foreach ($validated['mapel_heads'] ?? [] as $mapelHeadId => $data) {
                if (empty($data['nilai_akhir']) && empty($data['capaian_kompetensi'])) {
                    continue;
                }

                RaportDetail::create([
                    'raport_head_id' => $raportHead->id,
                    'mapel_head_id' => $mapelHeadId,
                    'mapel_detail_id' => null,
                    'nilai_akhir' => $data['nilai_akhir'] ?? null,
                    'deskripsi' => $data['capaian_kompetensi'] ?? null,
                ]);
            }

            foreach ($validated['mapel_details'] ?? [] as $mapelDetailId => $data) {
                if (empty($data['nilai_akhir']) && empty($data['capaian_kompetensi'])) {
                    continue;
                }

                RaportDetail::create([
                    'raport_head_id' => $raportHead->id,
                    'mapel_head_id' => null,
                    'mapel_detail_id' => $mapelDetailId,
                    'nilai_akhir' => $data['nilai_akhir'] ?? null,
                    'deskripsi' => $data['capaian_kompetensi'] ?? null,
                ]);
            }

            $raportHead->ekskuls()->update(['deleted' => '1']);

            foreach ($validated['ekstrakurikuler'] ?? [] as $row) {
                if (empty($row['kegiatan_id']) && empty($row['nilai_id']) && empty($row['keterangan'])) {
                    continue;
                }

                if (empty($row['kegiatan_id']) || empty($row['nilai_id'])) {
                    throw ValidationException::withMessages([
                        'ekstrakurikuler' => 'Setiap ekstrakurikuler harus memiliki kegiatan dan nilai.',
                    ]);
                }

                RaportEkskul::create([
                    'raport_head_id' => $raportHead->id,
                    'ekskul_id' => $row['kegiatan_id'],
                    'ket_nilai' => $row['nilai_id'] ?? null,
                    'keterangan' => $row['keterangan'] ?? null,
                ]);
            }

            $raportHead->prestasis()->update(['deleted' => '1']);

            foreach ($validated['prestasi'] ?? [] as $row) {
                if (empty($row['jenis']) && empty($row['keterangan'])) {
                    continue;
                }

                RaportPrestasi::create([
                    'raport_head_id' => $raportHead->id,
                    'jenis_prestasi' => $row['jenis'] ?? null,
                    'keterangan' => $row['keterangan'] ?? null,
                ]);
            }
        });

        return redirect()->route('raport.index')->with('successNotif', 'Raport berhasil diperbarui.');
    }

    /**
     * Download raport as PDF
     */
    public function download(Request $request, string $id)
    {
        Carbon::setLocale('id');

        $raportHead = RaportHead::with([
            'siswa.kelas',
            'details.mapelHead',
            'details.mapelDetail.mapelHead',
            'ekskuls.ekskul',
            'ekskuls.ketNilai',
            'prestasis',
            'kepalaSekolah',
            'waliKelas',
        ])
            ->where('deleted', '0')
            ->find($id);

        if ($raportHead) {
            $siswa = $raportHead->siswa;
            if (!$siswa) {
                abort(404, 'Data siswa untuk raport ini tidak ditemukan.');
            }
        } else {
            $siswa = Siswa::with('kelas')->findOrFail($id);

            $semesterFilter = $request->get('semester');

            $raportHead = RaportHead::with([
                'siswa.kelas',
                'details.mapelHead',
                'details.mapelDetail.mapelHead',
                'ekskuls.ekskul',
                'ekskuls.ketNilai',
                'prestasis',
                'kepalaSekolah',
                'waliKelas',
            ])
                ->where('siswa_id', $siswa->id)
                ->where('deleted', '0');

            if ($semesterFilter && in_array($semesterFilter, ['Ganjil', 'Genap'])) {
                $raportHead->where('semester', $semesterFilter);
            }

            $raportHead = $raportHead->latest('tanggal_titimangsa')->first();
        }

        if (!$raportHead) {
            abort(404, 'Data raport tidak ditemukan.');
        }

        $kelasId = $siswa->kelas_id;

        // Get mapel heads that are assigned to this kelas
        $mapelHeadIds = MapelKelas::where('kelas_id', $kelasId)
            ->where('mapel_type', 'head')
            ->pluck('mapel_id');

        // Get mapel detail ids that are assigned to this kelas
        $mapelDetailIds = MapelKelas::where('kelas_id', $kelasId)
            ->where('mapel_type', 'detail')
            ->pluck('mapel_id');

        $mapelHeads = \App\Models\MapelHead::with(['details' => function ($query) use ($mapelDetailIds) {
            $query->where('status', 'Aktif')
                ->where('deleted', '0')
                ->whereIn('id', $mapelDetailIds)
                ->orderBy('nama_mapel_detail');
        }])
            ->where('status', 'Aktif')
            ->where('deleted', '0')
            ->whereIn('id', $mapelHeadIds)
            ->orderBy('nama_mapel')
            ->get();

        $headScores = $raportHead->details
            ->whereNotNull('mapel_head_id')
            ->keyBy('mapel_head_id');

        $detailScores = $raportHead->details
            ->whereNotNull('mapel_detail_id')
            ->keyBy('mapel_detail_id');

        $ekskulPengisian = $raportHead->ekskuls;
        $prestasiPengisian = $raportHead->prestasis;

        $ketidakhadiran = [
            'sakit' => $raportHead->jml_sakit ?? 0,
            'izin' => $raportHead->jml_izin ?? 0,
            'alpa' => $raportHead->jml_alpa ?? 0,
        ];

        $formattedTanggal = $raportHead->tanggal_titimangsa
            ? Carbon::parse($raportHead->tanggal_titimangsa)->translatedFormat('d F Y')
            : null;

        $raportMapelHeadIds = $raportHead->details
            ->pluck('mapel_head_id')
            ->filter();

        $raportMapelHeadIds = $raportMapelHeadIds->merge(
            $raportHead->details
                ->pluck('mapelDetail.mapelHead.id')
                ->filter()
        )->unique();

        $missingHeadIds = $raportMapelHeadIds->diff($mapelHeads->pluck('id'));

        if ($missingHeadIds->isNotEmpty()) {
            $additionalHeads = \App\Models\MapelHead::with(['details' => function ($query) {
                $query->orderBy('nama_mapel_detail');
            }])
                ->whereIn('id', $missingHeadIds)
                ->get();

            $mapelHeads = $mapelHeads
                ->merge($additionalHeads)
                ->unique('id')
                ->sortBy('nama_mapel')
                ->values();
        }

        $kelas = $siswa->kelas;
        $roman = trim($kelas->nama_kelas_romawi ?? '');
        $sub = trim($kelas->nama_kelas_sub ?? '');
        $number = trim((string) ($kelas->nama_kelas_nomor ?? ''));

        $kelasDisplay = $roman;
        if ($sub !== '') {
            $kelasDisplay = trim($kelasDisplay . ' ' . $sub);
        }

        $numberWithSub = $number;
        if ($sub !== '') {
            $numberWithSub .= $sub;
        }

        if ($kelasDisplay && $numberWithSub) {
            $kelasDisplay;
        }

        $data = [
            'siswa' => $siswa,
            'raportHead' => $raportHead,
            'mapelHeads' => $mapelHeads,
            'headScores' => $headScores,
            'detailScores' => $detailScores,
            'ekskulPengisian' => $ekskulPengisian,
            'prestasiPengisian' => $prestasiPengisian,
            'ketidakhadiran' => $ketidakhadiran,
            'formattedTanggal' => $formattedTanggal,
            'kelasDisplay' => $kelasDisplay ?: '-',
        ];

        $pdf = Pdf::loadView('pages.raport.pdf', $data);
        $pdf->setPaper('a4', 'portrait');

        $filename = 'Raport_' . str_replace(' ', '_', $siswa->nama_siswa) . '_' . $raportHead->semester . '_' . $raportHead->thn_ajaran_awal . '-' . $raportHead->thn_ajaran_akhir . '.pdf';

        // return view('pages.raport.pdf', $data);
        return $pdf->download($filename);
    }

    /**
     * Download multiple raport as PDF (bulk download)
     */
    public function downloadBulk(Request $request)
    {
        $ids = $request->get('ids');
        $semester = $request->get('semester', 'all');

        if (empty($ids)) {
            abort(404, 'Tidak ada raport yang dipilih untuk diunduh.');
        }

        // Convert comma-separated string to array
        $raportHeadIds = explode(',', $ids);
        $raportHeadIds = array_filter($raportHeadIds);

        if (empty($raportHeadIds)) {
            abort(404, 'Tidak ada raport yang valid untuk diunduh.');
        }

        Carbon::setLocale('id');

        // Collect all raport data
        $allRaportData = [];

        foreach ($raportHeadIds as $id) {
            $raportHead = RaportHead::with([
                'siswa.kelas',
                'details.mapelHead',
                'details.mapelDetail.mapelHead',
                'ekskuls.ekskul',
                'ekskuls.ketNilai',
                'prestasis',
                'kepalaSekolah',
                'waliKelas',
            ])
                ->where('deleted', '0')
                ->find($id);

            if (!$raportHead || !$raportHead->siswa) {
                continue; // Skip if raport not found
            }

            $siswa = $raportHead->siswa;

            $kelasId = $siswa->kelas_id;

            // Get mapel heads that are assigned to this kelas
            $mapelHeadIds = MapelKelas::where('kelas_id', $kelasId)
                ->where('mapel_type', 'head')
                ->pluck('mapel_id');

            // Get mapel detail ids that are assigned to this kelas
            $mapelDetailIds = MapelKelas::where('kelas_id', $kelasId)
                ->where('mapel_type', 'detail')
                ->pluck('mapel_id');

            $mapelHeads = \App\Models\MapelHead::with(['details' => function ($query) use ($mapelDetailIds) {
                $query->where('status', 'Aktif')
                    ->where('deleted', '0')
                    ->whereIn('id', $mapelDetailIds)
                    ->orderBy('nama_mapel_detail');
            }])
                ->where('status', 'Aktif')
                ->where('deleted', '0')
                ->whereIn('id', $mapelHeadIds)
                ->orderBy('nama_mapel')
                ->get();

            $headScores = $raportHead->details
                ->whereNotNull('mapel_head_id')
                ->keyBy('mapel_head_id');

            $detailScores = $raportHead->details
                ->whereNotNull('mapel_detail_id')
                ->keyBy('mapel_detail_id');

            $ekskulPengisian = $raportHead->ekskuls;
            $prestasiPengisian = $raportHead->prestasis;

            $ketidakhadiran = [
                'sakit' => $raportHead->jml_sakit ?? 0,
                'izin' => $raportHead->jml_izin ?? 0,
                'alpa' => $raportHead->jml_alpa ?? 0,
            ];

            $formattedTanggal = $raportHead->tanggal_titimangsa
                ? Carbon::parse($raportHead->tanggal_titimangsa)->translatedFormat('d F Y')
                : null;

            $raportMapelHeadIds = $raportHead->details
                ->pluck('mapel_head_id')
                ->filter();

            $raportMapelHeadIds = $raportMapelHeadIds->merge(
                $raportHead->details
                    ->pluck('mapelDetail.mapelHead.id')
                    ->filter()
            )->unique();

            $missingHeadIds = $raportMapelHeadIds->diff($mapelHeads->pluck('id'));

            if ($missingHeadIds->isNotEmpty()) {
                $additionalHeads = \App\Models\MapelHead::with(['details' => function ($query) {
                    $query->orderBy('nama_mapel_detail');
                }])
                    ->whereIn('id', $missingHeadIds)
                    ->get();

                $mapelHeads = $mapelHeads
                    ->merge($additionalHeads)
                    ->unique('id')
                    ->sortBy('nama_mapel')
                    ->values();
            }

            $kelas = $siswa->kelas;
            $roman = trim($kelas->nama_kelas_romawi ?? '');
            $sub = trim($kelas->nama_kelas_sub ?? '');
            $number = trim((string) ($kelas->nama_kelas_nomor ?? ''));

            $kelasDisplay = $roman;
            if ($sub !== '') {
                $kelasDisplay = trim($kelasDisplay . ' ' . $sub);
            }

            $numberWithSub = $number;
            if ($sub !== '') {
                $numberWithSub .= $sub;
            }

            if ($kelasDisplay && $numberWithSub) {
                $kelasDisplay;
            }

            $allRaportData[] = [
                'siswa' => $siswa,
                'raportHead' => $raportHead,
                'mapelHeads' => $mapelHeads,
                'headScores' => $headScores,
                'detailScores' => $detailScores,
                'ekskulPengisian' => $ekskulPengisian,
                'prestasiPengisian' => $prestasiPengisian,
                'ketidakhadiran' => $ketidakhadiran,
                'formattedTanggal' => $formattedTanggal,
                'kelasDisplay' => $kelasDisplay ?: '-',
            ];
        }

        if (empty($allRaportData)) {
            abort(404, 'Tidak ada data raport yang valid untuk diunduh.');
        }

        // Generate single PDF with all raport data
        $data = [
            'raportList' => $allRaportData,
        ];

        $pdf = Pdf::loadView('pages.raport.pdf-bulk', $data);
        $pdf->setPaper('a4', 'portrait');

        // Get first raport info for filename
        $firstRaport = $allRaportData[0]['raportHead'];
        $semesterLabel = $semester === 'all' ? 'Semua_Semester' : 'Semester_' . $semester;
        $filename = 'Raport_' . $semesterLabel . '_' . $firstRaport->thn_ajaran_awal . '-' . $firstRaport->thn_ajaran_akhir . '_' . date('YmdHis') . '.pdf';

        return $pdf->download($filename);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
}
