<?php

namespace App\Http\Controllers;

use App\Http\Requests\ProfileUpdateRequest;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use Illuminate\View\View;
use App\Models\WaliKelas;

class ProfileController extends Controller
{
    /**
     * Display profil page
     */
    public function index()
    {
        $user = Auth::user();
        $waliKelas = null;

        // Jika user adalah wali kelas, ambil data wali kelas
        if ($user->role === 'Wali') {
            $waliKelas = WaliKelas::where('user_id', $user->id)
                ->where('deleted', '0')
                ->first();
        }

        return view('pages.profil.index', compact('user', 'waliKelas'));
    }

    /**
     * Update profil user
     */
    public function updateProfile(Request $request)
    {
        try {
            $user = Auth::user();

            // Determine if logo is required (required if user doesn't have logo yet and not removing)
            $logoRule = 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048';
            if (!$user->logo && !$request->has('remove_logo')) {
                $logoRule = 'required|image|mimes:jpeg,png,jpg,gif|max:2048';
            }

            $validated = $request->validate([
                'name' => 'required|string|max:255',
                'email' => 'required|string|max:255|unique:users,email,' . $user->id,
                'nama_sekolah' => 'required|string|max:100',
                'alamat_sekolah' => 'required|string|max:255',
                'logo' => $logoRule,
                'remove_logo' => 'nullable|boolean',
                'no_hp' => 'nullable|string|max:20',
                'email_wali' => 'nullable|email|max:255',
                'alamat' => 'nullable|string',
            ]);

            // Handle logo upload
            if ($request->hasFile('logo')) {
                // Delete old logo if exists
                if ($user->logo && file_exists(public_path('logo_app/' . $user->logo))) {
                    unlink(public_path('logo_app/' . $user->logo));
                }
                
                // Upload new logo
                $file = $request->file('logo');
                $filename = time() . '_' . preg_replace('/[^a-zA-Z0-9._-]/', '_', $file->getClientOriginalName());
                $file->move(public_path('logo_app'), $filename);
                $user->logo = $filename;
            }
            
            // Handle remove logo
            if ($request->has('remove_logo') && $request->remove_logo == '1') {
                if ($user->logo && file_exists(public_path('logo_app/' . $user->logo))) {
                    unlink(public_path('logo_app/' . $user->logo));
                }
                $user->logo = null;
            }

            // Update users table
            $user->email = $validated['email'];
            $user->name = $validated['name'];
            $user->nama_sekolah = $validated['nama_sekolah'] ?? $user->nama_sekolah;
            $user->alamat_sekolah = $validated['alamat_sekolah'] ?? $user->alamat_sekolah;
            $user->save();

            // Jika user adalah wali kelas, update juga data wali_kelas
            if ($user->role === 'Wali') {
                $waliKelas = WaliKelas::where('user_id', $user->id)
                    ->where('deleted', '0')
                    ->first();

                if ($waliKelas) {
                    // Nama lengkap juga update wali_kelas.nama_guru
                    $waliKelas->nama_guru = $validated['name'];
                    // No HP, Email, Alamat hanya update wali_kelas
                    $waliKelas->no_hp = $validated['no_hp'] ?? $waliKelas->no_hp;
                    $waliKelas->email = $validated['email_wali'] ?? $waliKelas->email;
                    $waliKelas->alamat = $validated['alamat'] ?? $waliKelas->alamat;
                    $waliKelas->save();
                }
            }

            return redirect()->back()->with('successNotif', 'Profil berhasil diperbarui');
        } catch (\Exception $e) {
            Log::error('Error updating profile: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Gagal memperbarui profil: ' . $e->getMessage());
        }
    }

    /**
     * Change password
     */
    public function changePassword(Request $request)
    {
        try {
            $validated = $request->validate([
                'current_password' => 'required',
                'new_password' => 'required|min:8|confirmed',
            ]);

            $user = Auth::user();

            // Check if current password is correct
            if (!Hash::check($validated['current_password'], $user->password)) {
                throw ValidationException::withMessages([
                    'current_password' => 'Password lama tidak sesuai',
                ]);
            }

            // Update password
            $user->password = Hash::make($validated['new_password']);
            $user->save();

            return redirect()->back()->with('successNotif', 'Password berhasil diubah');
        } catch (ValidationException $e) {
            return redirect()->back()->withErrors($e->errors())->withInput();
        } catch (\Exception $e) {
            Log::error('Error changing password: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Gagal mengubah password: ' . $e->getMessage());
        }
    }

    /**
     * Display the user's profile form.
     */
    public function edit(Request $request): View
    {
        return view('profile.edit', [
            'user' => $request->user(),
        ]);
    }

    /**
     * Update the user's profile information.
     */
    public function update(ProfileUpdateRequest $request): RedirectResponse
    {
        $request->user()->fill($request->validated());

        if ($request->user()->isDirty('email')) {
            $request->user()->email_verified_at = null;
        }

        $request->user()->save();

        return Redirect::route('profile.edit')->with('status', 'profile-updated');
    }

    /**
     * Delete the user's account.
     */
    public function destroy(Request $request): RedirectResponse
    {
        $request->validateWithBag('userDeletion', [
            'password' => ['required', 'current_password'],
        ]);

        $user = $request->user();

        Auth::logout();

        $user->delete();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return Redirect::to('/');
    }
}
