<?php

namespace App\Http\Controllers;

use App\Models\Pekerjaan;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class PekerjaanController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $perPage = $request->query('per_page', 20);
        $search = $request->input('search');

        // Get data for display
        $dataPekerjaan = Pekerjaan::when($search, function ($query) use ($search) {
            return $query->where('nama_pekerjaan', 'like', '%' . $search . '%');
        })
            ->where('deleted', '0')
            ->orderBy('nama_pekerjaan', 'asc')
            ->paginate($perPage)
            ->appends(request()->query());

        // Get pekerjaan for edit if edit parameter exists
        $pekerjaanEdit = null;
        if ($request->has('edit')) {
            $pekerjaanEdit = Pekerjaan::find($request->get('edit'));
        }

        return view('pages.pekerjaan.index', compact('dataPekerjaan', 'pekerjaanEdit'))
            ->with('i', (request()->input('page', 1) - 1) * $perPage);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return redirect()->route('pekerjaan.index');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validateData = $request->validate([
            'nama_pekerjaan' => 'required|max:50|unique:pekerjaans,nama_pekerjaan',
        ], [
            'nama_pekerjaan.required' => 'Nama pekerjaan harus diisi',
            'nama_pekerjaan.max' => 'Nama pekerjaan maksimal 50 karakter',
            'nama_pekerjaan.unique' => 'Nama pekerjaan sudah terdaftar',
        ]);

        try {
            Pekerjaan::create($validateData);
            return redirect()->route('pekerjaan.index')->with('successNotif', 'Data pekerjaan "' . $validateData['nama_pekerjaan'] . '" berhasil ditambahkan');
        } catch (\Throwable $th) {
            Log::error('Error saat menyimpan data pekerjaan: ' . $th->getMessage());
            return back()->with('error', 'Data pekerjaan gagal ditambahkan')->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        return redirect()->route('pekerjaan.index', ['edit' => $id]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $validateData = $request->validate([
            'nama_pekerjaan' => 'required|max:50|unique:pekerjaans,nama_pekerjaan,' . $id,
        ], [
            'nama_pekerjaan.required' => 'Nama pekerjaan harus diisi',
            'nama_pekerjaan.max' => 'Nama pekerjaan maksimal 50 karakter',
            'nama_pekerjaan.unique' => 'Nama pekerjaan sudah terdaftar',
        ]);

        try {
            $pekerjaan = Pekerjaan::findOrFail($id);
            $pekerjaan->update($validateData);
            return redirect()->route('pekerjaan.index')->with('successNotif', 'Data pekerjaan "' . $validateData['nama_pekerjaan'] . '" berhasil diperbarui');
        } catch (\Throwable $th) {
            Log::error('Error saat memperbarui data pekerjaan: ' . $th->getMessage());
            return back()->with('error', 'Data pekerjaan gagal diperbarui')->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
}
