<?php

namespace App\Http\Controllers;

use App\Models\MapelHead;
use App\Models\MapelDetail;
use App\Models\Kelas;
use App\Models\MapelKelas;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class MataPelajaranController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $perPage = $request->query('per_page', 20);
        $search = $request->input('search');

        $dataMapel = MapelHead::with([
            'details.mapelKelas' => function ($query) {
                $query->where('mapel_type', 'detail')->with('kelas');
            },
            'mapelKelas' => function ($query) {
                $query->where('mapel_type', 'head')->with('kelas');
            }
        ])
            ->when($search, function ($query) use ($search) {
                return $query->where('nama_mapel', 'like', '%' . $search . '%')
                    ->orWhereHas('details', function ($detailQuery) use ($search) {
                        $detailQuery->where('nama_mapel_detail', 'like', '%' . $search . '%')
                            ->where('deleted', '0');
                    });
            })
            ->where('deleted', '0')
            ->orderBy('nama_mapel', 'asc')
            ->paginate($perPage)
            ->appends(request()->query());

        $dataKelas = Kelas::where('deleted', '0')
            ->orderBy('nama_kelas_nomor', 'asc')
            ->orderBy('nama_kelas_sub', 'asc')
            ->get();

        return view('pages.mata_pelajaran.index', compact('dataMapel', 'dataKelas'))
            ->with('i', (request()->input('page', 1) - 1) * $perPage);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validateData = $request->validate([
            'nama_mapel'   => 'required|unique:mapel_heads,nama_mapel',
        ], [
            'nama_mapel.required' => 'Nama mata pelajaran harus diisi',
            'nama_mapel.unique'   => 'Nama mata pelajaran sudah terdaftar',
        ]);

        try {
            MapelHead::create($validateData);
            return back()->with('successNotif', 'Data mapel "' . $validateData['nama_mapel'] . '" berhasil ditambahkan');
        } catch (\Throwable $th) {
            Log::error('Error saat menyimpan data mapel: ' . $th->getMessage());
            return back()->with('error', 'Data mapel gagal ditambahkan');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $validateData = $request->validate([
            'nama_mapel'   => 'required|unique:mapel_heads,nama_mapel,' . $id,
        ], [
            'nama_mapel.required' => 'Nama mata pelajaran harus diisi',
            'nama_mapel.unique'   => 'Nama mata pelajaran sudah terdaftar',
        ]);

        // try {
        MapelHead::where('id', $id)->update([
            'nama_mapel' => $validateData['nama_mapel'],
            'status' => $request->status,
        ]);
        return back()->with('successNotif', 'Data mapel "' . $validateData['nama_mapel'] . '" berhasil diperbarui');
        // } catch (\Throwable $th) {
        //     Log::error('Error saat memperbarui data mapel: ' . $th->getMessage());
        //     return back()->with('error', 'Data mapel gagal diperbarui');
        // }
    }

    /**
     * Check if mapel head is being used in book_details
     */
    public function checkUsage(string $id)
    {
        $mapelHead = MapelHead::with('details')->findOrFail($id);
        
        // Check if mapel_head is used in book_details
        $usedInBookDetails = \App\Models\BookDetail::where('mapel_head_id', $id)
            ->where('deleted', '0')
            ->exists();
        
        if ($usedInBookDetails) {
            return response()->json([
                'used' => true,
                'nama_mapel' => $mapelHead->nama_mapel,
                'type' => 'head'
            ]);
        }
        
        // Check if any detail is used in book_details
        $detailIds = $mapelHead->details->pluck('id');
        $detailUsed = \App\Models\BookDetail::whereIn('mapel_detail_id', $detailIds)
            ->where('deleted', '0')
            ->exists();
        
        if ($detailUsed) {
            return response()->json([
                'used' => true,
                'nama_mapel' => $mapelHead->nama_mapel,
                'type' => 'head'
            ]);
        }
        
        return response()->json([
            'used' => false,
            'nama_mapel' => $mapelHead->nama_mapel
        ]);
    }

    /**
     * Check if mapel detail is being used in book_details
     */
    public function checkDetailUsage(string $id)
    {
        $mapelDetail = MapelDetail::with('mapelHead')->findOrFail($id);
        
        // Check if mapel_detail is used in book_details
        $usedInBookDetails = \App\Models\BookDetail::where('mapel_detail_id', $id)
            ->where('deleted', '0')
            ->exists();
        
        return response()->json([
            'used' => $usedInBookDetails,
            'nama_mapel' => $mapelDetail->nama_mapel_detail,
            'type' => 'detail'
        ]);
    }

    /**
     * Remove the specified resource from storage (Soft Delete MapelHead)
     */
    public function destroy(string $id)
    {
        try {
            $mapelHead = MapelHead::findOrFail($id);
            $namaMapel = $mapelHead->nama_mapel;
            
            // Soft delete mapel_head
            $mapelHead->update(['deleted' => '1']);
            
            // Soft delete all related mapel_details
            $detailsCount = MapelDetail::where('mapel_head_id', $id)
                ->where('deleted', '0')
                ->count();
            
            MapelDetail::where('mapel_head_id', $id)
                ->update(['deleted' => '1']);
            
            // Hard delete all related mapel_kelas (both head and details)
            $mapelKelasHeadCount = MapelKelas::where('mapel_id', $id)
                ->where('mapel_type', 'head')
                ->count();
            
            MapelKelas::where('mapel_id', $id)
                ->where('mapel_type', 'head')
                ->delete();
            
            // Delete mapel_kelas for all details
            $detailIds = MapelDetail::where('mapel_head_id', $id)->pluck('id');
            $mapelKelasDetailCount = MapelKelas::whereIn('mapel_id', $detailIds)
                ->where('mapel_type', 'detail')
                ->count();
            
            MapelKelas::whereIn('mapel_id', $detailIds)
                ->where('mapel_type', 'detail')
                ->delete();
            
            $message = 'Mata pelajaran "' . $namaMapel . '" berhasil dihapus';
            
            if ($detailsCount > 0) {
                $message .= ' beserta ' . $detailsCount . ' detail mapel';
            }
            
            $totalMapelKelas = $mapelKelasHeadCount + $mapelKelasDetailCount;
            if ($totalMapelKelas > 0) {
                $message .= ' dan ' . $totalMapelKelas . ' relasi kelas';
            }
            
            return back()->with('successNotif', $message);
        } catch (\Throwable $th) {
            Log::error('Error saat menghapus mapel head: ' . $th->getMessage());
            return back()->with('error', 'Gagal menghapus mata pelajaran: ' . $th->getMessage());
        }
    }

    /**
     * Remove mapel detail from storage (Soft Delete)
     */
    public function destroyDetail(string $id)
    {
        try {
            $mapelDetail = MapelDetail::findOrFail($id);
            $namaMapelDetail = $mapelDetail->nama_mapel_detail;
            
            // Soft delete mapel_detail
            $mapelDetail->update(['deleted' => '1']);
            
            // Hard delete related mapel_kelas
            $mapelKelasCount = MapelKelas::where('mapel_id', $id)
                ->where('mapel_type', 'detail')
                ->count();
            
            MapelKelas::where('mapel_id', $id)
                ->where('mapel_type', 'detail')
                ->delete();
            
            $message = 'Detail mata pelajaran "' . $namaMapelDetail . '" berhasil dihapus';
            
            if ($mapelKelasCount > 0) {
                $message .= ' beserta ' . $mapelKelasCount . ' relasi kelas';
            }
            
            return back()->with('successNotif', $message);
        } catch (\Throwable $th) {
            Log::error('Error saat menghapus mapel detail: ' . $th->getMessage());
            return back()->with('error', 'Gagal menghapus detail mata pelajaran: ' . $th->getMessage());
        }
    }

    /**
     * Store mapel detail
     */
    public function storeDetail(Request $request)
    {
        $validateData = $request->validate([
            'mapel_head_id' => 'required|exists:mapel_heads,id',
            'nama_mapel_detail' => 'required|unique:mapel_details,nama_mapel_detail|max:50',
        ], [
            'mapel_head_id.required' => 'Mata pelajaran harus dipilih',
            'nama_mapel_detail.required' => 'Nama detail mata pelajaran harus diisi',
            'nama_mapel_detail.unique' => 'Nama detail mata pelajaran sudah terdaftar',
        ]);

        try {
            MapelDetail::create(
                [
                    'mapel_head_id' => $validateData['mapel_head_id'],
                    'nama_mapel_detail' => $validateData['nama_mapel_detail'],
                    'status' => 'Aktif',
                ]
            );
            return back()->with('successNotif', 'Detail mata pelajaran "' . $validateData['nama_mapel_detail'] . '" berhasil ditambahkan');
        } catch (\Throwable $th) {
            Log::error('Error saat menyimpan detail mapel: ' . $th->getMessage());
            return back()->with('error', 'Detail mata pelajaran gagal ditambahkan');
        }
    }

    /**
     * Update mapel detail
     */
    public function updateDetail(Request $request, string $id)
    {
        $validateData = $request->validate([
            'nama_mapel_detail' => 'required|max:50|unique:mapel_details,nama_mapel_detail,' . $id,
            'status' => 'required|in:Aktif,Non-Aktif',
        ], [
            'nama_mapel_detail.required' => 'Nama detail mata pelajaran harus diisi',
            'nama_mapel_detail.unique' => 'Nama detail mata pelajaran sudah terdaftar',
            'status.required' => 'Status harus dipilih',
        ]);

        try {
            MapelDetail::where('id', $id)->update($validateData);
            return back()->with('successNotif', 'Detail mata pelajaran "' . $validateData['nama_mapel_detail'] . '" berhasil diperbarui');
        } catch (\Throwable $th) {
            Log::error('Error saat memperbarui detail mapel: ' . $th->getMessage());
            return back()->with('error', 'Detail mata pelajaran gagal diperbarui');
        }
    }

    /**
     * Store kelas for mapel (head or detail)
     */
    public function storeKelas(Request $request)
    {
        $validateData = $request->validate([
            'mapel_id' => 'required',
            'mapel_type' => 'required|in:head,detail',
            'kelas_id' => 'array',
            'kelas_id.*' => 'exists:kelas,id',
        ], [
            'mapel_id.required' => 'Mata pelajaran harus dipilih',
            'kelas_id.array' => 'Kelas harus berupa array',
        ]);

        try {
            $mapelId = $validateData['mapel_id'];
            $mapelType = $validateData['mapel_type'];
            $kelasIds = $request->input('kelas_id', []);

            // Delete existing records for this mapel with specific type
            MapelKelas::where('mapel_id', $mapelId)
                ->where('mapel_type', $mapelType)
                ->delete();

            // Insert new records only if there are kelas selected
            if (!empty($kelasIds)) {
                foreach ($kelasIds as $kelasId) {
                    MapelKelas::create([
                        'mapel_id' => $mapelId,
                        'mapel_type' => $mapelType,
                        'kelas_id' => $kelasId,
                    ]);
                }
            }

            // Get mapel name for notification
            if ($mapelType === 'head') {
                $mapel = MapelHead::find($mapelId);
                $mapelName = $mapel->nama_mapel;
            } else {
                $mapel = MapelDetail::find($mapelId);
                $mapelName = $mapel->nama_mapel_detail;
            }

            $message = !empty($kelasIds) 
                ? 'Kelas untuk "' . $mapelName . '" berhasil diperbarui' 
                : 'Semua kelas untuk "' . $mapelName . '" berhasil dihapus';

            return back()->with('successNotif', $message);
        } catch (\Throwable $th) {
            Log::error('Error saat menyimpan kelas mapel: ' . $th->getMessage());
            return back()->with('error', 'Kelas gagal diperbarui: ' . $th->getMessage());
        }
    }
}
