<?php

namespace App\Http\Controllers;

use App\Models\KepalaSekolah;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class KepalaSekolahController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $perPage = $request->query('per_page', 20);
        $search = $request->input('search');

        // Get data for display
        $dataKepalaSekolah = KepalaSekolah::when($search, function ($query) use ($search) {
            return $query->where('nama_lengkap', 'like', '%' . $search . '%')
                ->orWhere('nip', 'like', '%' . $search . '%');
        })
            ->where('deleted', '0')
            ->orderBy('nama_lengkap', 'asc')
            ->paginate($perPage)
            ->appends(request()->query());

        // Get kepala sekolah for edit if edit parameter exists
        $kepalaSekolahEdit = null;
        if ($request->has('edit')) {
            $kepalaSekolahEdit = KepalaSekolah::find($request->get('edit'));
        }

        return view('pages.kepala_sekolah.index', compact('dataKepalaSekolah', 'kepalaSekolahEdit'))
            ->with('i', (request()->input('page', 1) - 1) * $perPage);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return redirect()->route('kepala-sekolah.index');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validateData = $request->validate([
            'nama_lengkap' => 'required|max:50',
            'nip' => 'nullable|max:30|unique:kepala_sekolahs,nip',
            'status' => 'required|in:Aktif,Non-Aktif',
        ], [
            'nama_lengkap.required' => 'Nama lengkap harus diisi',
            'nama_lengkap.max' => 'Nama lengkap maksimal 50 karakter',
            'nip.max' => 'NIP maksimal 30 karakter',
            'nip.unique' => 'NIP sudah terdaftar',
            'status.required' => 'Status harus dipilih',
            'status.in' => 'Status tidak valid',
        ]);

        try {
            KepalaSekolah::create($validateData);
            
            $message = 'Data kepala sekolah "' . $validateData['nama_lengkap'] . '" berhasil ditambahkan';
            if ($validateData['status'] === 'Aktif') {
                $message .= ' dan status kepala sekolah lainnya diubah menjadi Non-Aktif';
            }
            
            return redirect()->route('kepala-sekolah.index')->with('successNotif', $message);
        } catch (\Throwable $th) {
            Log::error('Error saat menyimpan data kepala sekolah: ' . $th->getMessage());
            return back()->with('error', 'Data kepala sekolah gagal ditambahkan')->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        return redirect()->route('kepala-sekolah.index', ['edit' => $id]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $validateData = $request->validate([
            'nama_lengkap' => 'required|max:50',
            'nip' => 'nullable|max:30|unique:kepala_sekolahs,nip,' . $id,
            'status' => 'required|in:Aktif,Non-Aktif',
        ], [
            'nama_lengkap.required' => 'Nama lengkap harus diisi',
            'nama_lengkap.max' => 'Nama lengkap maksimal 50 karakter',
            'nip.max' => 'NIP maksimal 30 karakter',
            'nip.unique' => 'NIP sudah terdaftar',
            'status.required' => 'Status harus dipilih',
            'status.in' => 'Status tidak valid',
        ]);

        try {
            $kepalaSekolah = KepalaSekolah::findOrFail($id);
            $kepalaSekolah->update($validateData);
            
            $message = 'Data kepala sekolah "' . $validateData['nama_lengkap'] . '" berhasil diperbarui';
            if ($validateData['status'] === 'Aktif') {
                $message .= ' dan status kepala sekolah lainnya diubah menjadi Non-Aktif';
            }
            
            return redirect()->route('kepala-sekolah.index')->with('successNotif', $message);
        } catch (\Throwable $th) {
            Log::error('Error saat memperbarui data kepala sekolah: ' . $th->getMessage());
            return back()->with('error', 'Data kepala sekolah gagal diperbarui')->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
}
