<?php

namespace App\Http\Controllers;

use App\Models\Kelas;
use App\Models\WaliKelas;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Validator;

class KelasController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $perPage = $request->query('per_page', 20);
        $search = $request->input('search');

        // Get data for display
        $dataKelas = Kelas::when($search, function ($query) use ($search) {
            return $query->where(function ($q) use ($search) {
                $q->where('nama_kelas_nomor', 'like', '%' . $search . '%')
                    ->orWhere('nama_kelas_romawi', 'like', '%' . $search . '%')
                    ->orWhere('nama_kelas_sub', 'like', '%' . $search . '%');
            });
        })
            ->where('deleted', '0')
            ->orderBy('nama_kelas_nomor', 'asc')
            ->orderBy('nama_kelas_sub', 'asc')
            ->paginate($perPage)
            ->appends(request()->query());

        // Get kelas for edit if edit parameter exists
        $kelasEdit = null;
        if ($request->has('edit')) {
            $kelasEdit = Kelas::find($request->get('edit'));
        }

        return view('pages.kelas.index', compact('dataKelas', 'kelasEdit'))
            ->with('i', (request()->input('page', 1) - 1) * $perPage);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return redirect()->route('kelas.index');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'nama_kelas_nomor' => [
                'required',
                Rule::unique('kelas')->where(function ($query) use ($request) {
                    if (!empty($request->nama_kelas_sub)) {
                        return $query->where('nama_kelas_sub', $request->nama_kelas_sub);
                    }
                    return $query->whereNull('nama_kelas_sub');
                }),
            ],
            'nama_kelas_romawi' => 'required',
            'nama_kelas_sub' => 'nullable',
        ], [
            'nama_kelas_nomor.required' => 'Nama kelas harus diisi',
            'nama_kelas_romawi.required' => 'Nama kelas romawi harus diisi',
            'nama_kelas_nomor.unique' => 'Data tersebut sudah terdaftar',
        ]);

        // Tambahkan error ke 3 field jika kombinasi duplikat
        if ($validator->fails()) {
            if ($validator->errors()->has('nama_kelas_nomor')) {
                $message = 'Data tersebut sudah terdaftar';
                $validator->errors()->add('nama_kelas_romawi', $message);
                $validator->errors()->add('nama_kelas_sub', $message);
            }
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $validateData = $validator->validated();

        try {
            Kelas::create([
                'nama_kelas_nomor' => $validateData['nama_kelas_nomor'],
                'nama_kelas_romawi' => $validateData['nama_kelas_romawi'],
                'nama_kelas_sub' => $validateData['nama_kelas_sub'] ?? null,
            ]);
            return redirect()->route('kelas.index')->with(
                'successNotif',
                'Data kelas "' . $validateData['nama_kelas_romawi'] .
                    (!empty($validateData['nama_kelas_sub']) ? ' ' . strtoupper($validateData['nama_kelas_sub']) : '') .
                    '" berhasil ditambahkan'
            );
        } catch (\Throwable $th) {
            Log::error('Error saat menyimpan data kelas: ' . $th->getMessage());
            return back()->with('error', 'Data kelas gagal ditambahkan')->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        return redirect()->route('kelas.index', ['edit' => $id]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $kelas = Kelas::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'nama_kelas_nomor' => [
                'required',
                Rule::unique('kelas')->ignore($kelas->id)->where(function ($query) use ($request) {
                    if (!empty($request->nama_kelas_sub)) {
                        return $query->where('nama_kelas_sub', $request->nama_kelas_sub);
                    }
                    return $query->whereNull('nama_kelas_sub');
                }),
            ],
            'nama_kelas_romawi' => 'required',
            'nama_kelas_sub' => 'nullable',
        ], [
            'nama_kelas_nomor.required' => 'Nama kelas harus diisi',
            'nama_kelas_romawi.required' => 'Nama kelas romawi harus diisi',
            'nama_kelas_nomor.unique' => 'Data tersebut sudah terdaftar',
        ]);

        // Tambahkan error ke 3 field jika kombinasi duplikat
        if ($validator->fails()) {
            if ($validator->errors()->has('nama_kelas_nomor')) {
                $message = 'Data tersebut sudah terdaftar';
                $validator->errors()->add('nama_kelas_romawi', $message);
                $validator->errors()->add('nama_kelas_sub', $message);
            }
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $validateData = $validator->validated();

        try {
            $kelas->update([
                'nama_kelas_nomor' => $validateData['nama_kelas_nomor'],
                'nama_kelas_romawi' => $validateData['nama_kelas_romawi'],
                'nama_kelas_sub' => $validateData['nama_kelas_sub'] ?? null,
            ]);

            return redirect()->route('kelas.index')->with(
                'successNotif',
                'Data kelas "' . $validateData['nama_kelas_romawi'] .
                    (!empty($validateData['nama_kelas_sub']) ? ' ' . strtoupper($validateData['nama_kelas_sub']) : '') .
                    '" berhasil diperbarui'
            );
        } catch (\Throwable $th) {
            Log::error('Error saat memperbarui data kelas: ' . $th->getMessage());
            return back()->with('error', 'Data kelas gagal diperbarui')->withInput();
        }
    }

    /**
     * Check if kelas is being used
     */
    public function checkUsage(string $id)
    {
        try {
            $kelas = Kelas::findOrFail($id);
            
            // Build the full class name (e.g., "7A" or "7")
            $namaKelasLengkap = $kelas->nama_kelas_nomor . ($kelas->nama_kelas_sub ?? '');
            
            // Check if used in siswas table (by kelas_id)
            $usedInSiswas = \App\Models\Siswa::where('kelas_id', $kelas->id)
                ->where('deleted', '0')
                ->exists();
            
            // Check if used in book_heads table (by nama_kelas_nomor)
            $usedInBookHeads = \App\Models\BookHead::where('nama_kelas_nomor', $namaKelasLengkap)
                ->where('deleted', '0')
                ->exists();
            
            $displayName = $kelas->nama_kelas_romawi . ($kelas->nama_kelas_sub ?? '');
            
            return response()->json([
                'used' => $usedInSiswas || $usedInBookHeads,
                'nama_kelas' => $displayName
            ]);
        } catch (\Throwable $th) {
            return response()->json([
                'error' => 'Terjadi kesalahan saat memeriksa data'
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        try {
            $kelas = Kelas::findOrFail($id);
            
            // Build the full class name (e.g., "7A" or "7")
            $namaKelasLengkap = $kelas->nama_kelas_nomor . ($kelas->nama_kelas_sub ?? '');
            
            // Check if used in siswas or book_heads
            $usedInSiswas = \App\Models\Siswa::where('kelas_id', $kelas->id)
                ->where('deleted', '0')
                ->exists();
            
            $usedInBookHeads = \App\Models\BookHead::where('nama_kelas_nomor', $namaKelasLengkap)
                ->where('deleted', '0')
                ->exists();
            
            $displayName = $kelas->nama_kelas_romawi . ($kelas->nama_kelas_sub ?? '');
            
            if ($usedInSiswas || $usedInBookHeads) {
                return back()->with('error', 'Data kelas "' . $displayName . '" tidak dapat dihapus karena sudah digunakan');
            }
            
            // Soft delete
            $kelas->update(['deleted' => '1']);
            
            return redirect()->route('kelas.index')->with('successNotif', 'Data kelas "' . $displayName . '" berhasil dihapus');
        } catch (\Throwable $th) {
            Log::error('Error saat menghapus data kelas: ' . $th->getMessage());
            return back()->with('error', 'Data kelas gagal dihapus');
        }
    }
}
