<?php

namespace App\Http\Controllers;

use App\Models\Ekskuls;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class EkskulController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $perPage = $request->query('per_page', 20);
        $search = $request->input('search');

        // Get data for display
        $dataEkskul = Ekskuls::when($search, function ($query) use ($search) {
            return $query->where('nama_ekskul', 'like', '%' . $search . '%');
        })
            ->where('deleted', '0')
            ->orderBy('nama_ekskul', 'asc')
            ->paginate($perPage)
            ->appends(request()->query());

        // Get ekskul for edit if edit parameter exists
        $ekskulEdit = null;
        if ($request->has('edit')) {
            $ekskulEdit = Ekskuls::find($request->get('edit'));
        }

        return view('pages.ekstrakulikuler.index', compact('dataEkskul', 'ekskulEdit'))
            ->with('i', (request()->input('page', 1) - 1) * $perPage);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return redirect()->route('ekstrakulikuler.index');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validateData = $request->validate([
            'nama_ekskul' => 'required|max:50|unique:ekskuls,nama_ekskul',
            'status' => 'required|in:Aktif,Non-Aktif',
        ], [
            'nama_ekskul.required' => 'Nama ekstrakulikuler harus diisi',
            'nama_ekskul.max' => 'Nama ekstrakulikuler maksimal 50 karakter',
            'nama_ekskul.unique' => 'Nama ekstrakulikuler sudah terdaftar',
            'status.required' => 'Status harus dipilih',
            'status.in' => 'Status tidak valid',
        ]);

        try {
            Ekskuls::create($validateData);
            return redirect()->route('ekstrakulikuler.index')->with('successNotif', 'Data ekstrakulikuler "' . $validateData['nama_ekskul'] . '" berhasil ditambahkan');
        } catch (\Throwable $th) {
            Log::error('Error saat menyimpan data ekstrakulikuler: ' . $th->getMessage());
            return back()->with('error', 'Data ekstrakulikuler gagal ditambahkan')->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        return redirect()->route('ekstrakulikuler.index', ['edit' => $id]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $validateData = $request->validate([
            'nama_ekskul' => 'required|max:50|unique:ekskuls,nama_ekskul,' . $id,
            'status' => 'required|in:Aktif,Non-Aktif',
        ], [
            'nama_ekskul.required' => 'Nama ekstrakulikuler harus diisi',
            'nama_ekskul.max' => 'Nama ekstrakulikuler maksimal 50 karakter',
            'nama_ekskul.unique' => 'Nama ekstrakulikuler sudah terdaftar',
            'status.required' => 'Status harus dipilih',
            'status.in' => 'Status tidak valid',
        ]);

        try {
            $ekskul = Ekskuls::findOrFail($id);
            $ekskul->update($validateData);
            return redirect()->route('ekstrakulikuler.index')->with('successNotif', 'Data ekstrakulikuler "' . $validateData['nama_ekskul'] . '" berhasil diperbarui');
        } catch (\Throwable $th) {
            Log::error('Error saat memperbarui data ekstrakulikuler: ' . $th->getMessage());
            return back()->with('error', 'Data ekstrakulikuler gagal diperbarui')->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
}
