<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Siswa;
use App\Models\Kelas;
use App\Models\RaportHead;
use App\Models\BookHead;
use App\Models\BookDetail;
use App\Models\PeriodeWaliKelas;
use App\Models\WaliKelas;
use App\Models\TahunAjaran;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    public function index(Request $request)
    {
        // Get active tahun ajaran from TahunAjaran model
        $tahunAjaranAktif = TahunAjaran::where('status', 'Aktif')
            ->where('deleted', '0')
            ->first();

        // If no active tahun ajaran found, use current year as fallback
        if ($tahunAjaranAktif) {
            $thnAjaranAwal = $tahunAjaranAktif->tahun_awal;
            $thnAjaranAkhir = $tahunAjaranAktif->tahun_akhir;
            $selectedTahunAjaran = $thnAjaranAwal . '/' . $thnAjaranAkhir;
        } else {
            $tahunSekarang = date('Y');
            $thnAjaranAwal = $tahunSekarang;
            $thnAjaranAkhir = $tahunSekarang + 1;
            $selectedTahunAjaran = $thnAjaranAwal . '/' . $thnAjaranAkhir;
        }

        // Card 1: Total siswa aktif per kelas
        $siswaPerKelas = Siswa::with('kelas')
            ->where('thn_ajaran_awal', $thnAjaranAwal)
            ->where('thn_ajaran_akhir', $thnAjaranAkhir)
            ->where('status', 'Aktif')
            ->where('deleted', '0')
            ->get()
            ->groupBy('kelas_id')
            ->map(function ($students, $kelasId) {
                $kelas = $students->first()->kelas;

                if (!$kelas) {
                    return null;
                }

                $roman = trim($kelas->nama_kelas_romawi ?? '');
                $sub = trim($kelas->nama_kelas_sub ?? '');
                $number = trim((string) ($kelas->nama_kelas_nomor ?? ''));

                $romanWithSub = $roman;
                if ($sub !== '') {
                    $romanWithSub = trim($romanWithSub . ' ' . $sub);
                }

                $numberWithSub = $number;
                if ($sub !== '') {
                    $numberWithSub .= $sub;
                }

                $kelasDisplay = $romanWithSub;
                if ($romanWithSub !== '' && $numberWithSub !== '') {
                    $kelasDisplay .= ' (' . $numberWithSub . ')';
                }

                return [
                    'kelas_display' => $kelasDisplay ?: $kelas->nama_kelas,
                    'kelas_nomor' => $kelas->nama_kelas_nomor,
                    'kelas_sub' => $kelas->nama_kelas_sub,
                    'total' => $students->count(),
                ];
            })
            ->filter()
            ->sortBy([
                ['kelas_nomor', 'asc'],
                ['kelas_sub', 'asc'],
            ])
            ->values();

        // Card 2: Progress input raport per wali kelas (terpisah per semester)
        $waliKelasProgress = PeriodeWaliKelas::with(['waliKelas.user', 'kelas'])
            ->where('thn_ajaran_awal', $thnAjaranAwal)
            ->where('thn_ajaran_akhir', $thnAjaranAkhir)
            ->where('deleted', '0')
            ->get()
            ->map(function ($periode) use ($thnAjaranAwal, $thnAjaranAkhir) {
                // Get total students in this class
                $totalSiswa = Siswa::where('kelas_id', $periode->kelas_id)
                    ->where('thn_ajaran_awal', $thnAjaranAwal)
                    ->where('thn_ajaran_akhir', $thnAjaranAkhir)
                    ->where('status', 'Aktif')
                    ->where('deleted', '0')
                    ->count();

                // Count raport Ganjil
                $totalSudahInputGanjil = RaportHead::where('wali_kelas_id', $periode->wali_kelas_id)
                    ->where('semester', 'Ganjil')
                    ->where('thn_ajaran_awal', $thnAjaranAwal)
                    ->where('thn_ajaran_akhir', $thnAjaranAkhir)
                    ->where('deleted', '0')
                    ->whereIn('siswa_id', function ($query) use ($periode, $thnAjaranAwal, $thnAjaranAkhir) {
                        $query->select('id')
                            ->from('siswas')
                            ->where('kelas_id', $periode->kelas_id)
                            ->where('thn_ajaran_awal', $thnAjaranAwal)
                            ->where('thn_ajaran_akhir', $thnAjaranAkhir)
                            ->where('status', 'Aktif')
                            ->where('deleted', '0');
                    })
                    ->count();

                // Count raport Genap
                $totalSudahInputGenap = RaportHead::where('wali_kelas_id', $periode->wali_kelas_id)
                    ->where('semester', 'Genap')
                    ->where('thn_ajaran_awal', $thnAjaranAwal)
                    ->where('thn_ajaran_akhir', $thnAjaranAkhir)
                    ->where('deleted', '0')
                    ->whereIn('siswa_id', function ($query) use ($periode, $thnAjaranAwal, $thnAjaranAkhir) {
                        $query->select('id')
                            ->from('siswas')
                            ->where('kelas_id', $periode->kelas_id)
                            ->where('thn_ajaran_awal', $thnAjaranAwal)
                            ->where('thn_ajaran_akhir', $thnAjaranAkhir)
                            ->where('status', 'Aktif')
                            ->where('deleted', '0');
                    })
                    ->count();

                // Format kelas display
                $kelas = $periode->kelas;
                $roman = trim($kelas->nama_kelas_romawi ?? '');
                $sub = trim($kelas->nama_kelas_sub ?? '');
                $number = trim((string) ($kelas->nama_kelas_nomor ?? ''));

                $romanWithSub = $roman;
                if ($sub !== '') {
                    $romanWithSub = trim($romanWithSub . ' ' . $sub);
                }

                $numberWithSub = $number;
                if ($sub !== '') {
                    $numberWithSub .= $sub;
                }

                $kelasDisplay = $romanWithSub;
                if ($romanWithSub !== '' && $numberWithSub !== '') {
                    $kelasDisplay .= ' (' . $numberWithSub . ')';
                }

                return [
                    'wali_kelas_nama' => $periode->waliKelas->user->name ?? 'N/A',
                    'kelas_display' => $kelasDisplay ?: $kelas->nama_kelas,
                    'kelas_nomor' => $kelas->nama_kelas_nomor,
                    'kelas_sub' => $kelas->nama_kelas_sub,
                    'total_siswa' => $totalSiswa,
                    'ganjil' => [
                        'sudah_input' => $totalSudahInputGanjil,
                        'harus_input' => $totalSiswa,
                        'progress_percentage' => $totalSiswa > 0 ? round(($totalSudahInputGanjil / $totalSiswa) * 100, 1) : 0,
                    ],
                    'genap' => [
                        'sudah_input' => $totalSudahInputGenap,
                        'harus_input' => $totalSiswa,
                        'progress_percentage' => $totalSiswa > 0 ? round(($totalSudahInputGenap / $totalSiswa) * 100, 1) : 0,
                    ],
                ];
            })
            ->sortBy([
                ['kelas_nomor', 'asc'],
                ['kelas_sub', 'asc'],
            ])
            ->values();

        // Card 3: Ranking siswa per kelas (semua siswa, terpisah per semester)
        $user = Auth::user();
        $kelasQuery = Kelas::where('deleted', '0');

        // Filter kelas berdasarkan role user
        if ($user->role === 'Wali') {
            // Jika user adalah Wali, hanya tampilkan kelas yang diampu
            $waliKelas = WaliKelas::where('user_id', $user->id)
                ->where('deleted', '0')
                ->first();

            if ($waliKelas) {
                $kelasIds = PeriodeWaliKelas::where('wali_kelas_id', $waliKelas->id)
                    ->where('thn_ajaran_awal', $thnAjaranAwal)
                    ->where('thn_ajaran_akhir', $thnAjaranAkhir)
                    ->where('deleted', '0')
                    ->pluck('kelas_id');

                $kelasQuery->whereIn('id', $kelasIds);
            } else {
                // Jika tidak ada data wali kelas, return empty collection
                $kelasQuery->whereRaw('1 = 0');
            }
        }

        $rankingPerKelas = $kelasQuery
            ->orderBy('nama_kelas_nomor', 'asc')
            ->orderBy('nama_kelas_sub', 'asc')
            ->get()
            ->map(function ($kelas) use ($thnAjaranAwal, $thnAjaranAkhir) {
                // Get all students in this class
                $allSiswa = Siswa::where('kelas_id', $kelas->id)
                    ->where('thn_ajaran_awal', $thnAjaranAwal)
                    ->where('thn_ajaran_akhir', $thnAjaranAkhir)
                    ->where('deleted', '0')
                    ->orderBy('nama_siswa', 'asc')
                    ->get();

                if ($allSiswa->isEmpty()) {
                    return null;
                }

                // Get book data for Ganjil semester
                $raportGanjil = BookHead::whereIn('siswa_id', $allSiswa->pluck('id'))
                    ->where('semester', 'Ganjil')
                    ->where('thn_ajaran_awal', $thnAjaranAwal)
                    ->where('thn_ajaran_akhir', $thnAjaranAkhir)
                    ->where('deleted', '0')
                    ->orderBy('jml_nilai_akhir', 'desc')
                    ->get()
                    ->keyBy('siswa_id');

                // Get book data for Genap semester
                $raportGenap = BookHead::whereIn('siswa_id', $allSiswa->pluck('id'))
                    ->where('semester', 'Genap')
                    ->where('thn_ajaran_awal', $thnAjaranAwal)
                    ->where('thn_ajaran_akhir', $thnAjaranAkhir)
                    ->where('deleted', '0')
                    ->orderBy('jml_nilai_akhir', 'desc')
                    ->get()
                    ->keyBy('siswa_id');

                // Build ranking for Ganjil
                $rankingGanjil = $allSiswa->map(function ($siswa) use ($raportGanjil) {
                    $raport = $raportGanjil->get($siswa->id);
                    return [
                        'siswa_id' => $siswa->id,
                        'nama' => $siswa->nama_siswa,
                        'nis' => $siswa->nis,
                        'jml_nilai_akhir' => $raport ? $raport->jml_nilai_akhir : null,
                        'has_raport' => $raport !== null,
                    ];
                })->sortByDesc(function ($item) {
                    return $item['jml_nilai_akhir'] ?? -1;
                })->values();

                // Add rank to Ganjil
                $rank = 1;
                $rankingGanjil = $rankingGanjil->map(function ($item) use (&$rank) {
                    if ($item['has_raport']) {
                        $item['rank'] = $rank++;
                    } else {
                        $item['rank'] = null;
                    }
                    return $item;
                });

                // Build ranking for Genap
                $rankingGenap = $allSiswa->map(function ($siswa) use ($raportGenap) {
                    $raport = $raportGenap->get($siswa->id);
                    return [
                        'siswa_id' => $siswa->id,
                        'nama' => $siswa->nama_siswa,
                        'nis' => $siswa->nis,
                        'jml_nilai_akhir' => $raport ? $raport->jml_nilai_akhir : null,
                        'has_raport' => $raport !== null,
                    ];
                })->sortByDesc(function ($item) {
                    return $item['jml_nilai_akhir'] ?? -1;
                })->values();

                // Add rank to Genap
                $rank = 1;
                $rankingGenap = $rankingGenap->map(function ($item) use (&$rank) {
                    if ($item['has_raport']) {
                        $item['rank'] = $rank++;
                    } else {
                        $item['rank'] = null;
                    }
                    return $item;
                });

                // Format kelas display
                $roman = trim($kelas->nama_kelas_romawi ?? '');
                $sub = trim($kelas->nama_kelas_sub ?? '');
                $number = trim((string) ($kelas->nama_kelas_nomor ?? ''));

                $romanWithSub = $roman;
                if ($sub !== '') {
                    $romanWithSub = trim($romanWithSub . ' ' . $sub);
                }

                $numberWithSub = $number;
                if ($sub !== '') {
                    $numberWithSub .= $sub;
                }

                $kelasDisplay = $romanWithSub;
                if ($romanWithSub !== '' && $numberWithSub !== '') {
                    $kelasDisplay .= ' (' . $numberWithSub . ')';
                }

                return [
                    'kelas_id' => $kelas->id,
                    'kelas_display' => $kelasDisplay ?: $kelas->nama_kelas,
                    'kelas_nomor' => $kelas->nama_kelas_nomor,
                    'kelas_sub' => $kelas->nama_kelas_sub,
                    'ranking_ganjil' => $rankingGanjil,
                    'ranking_genap' => $rankingGenap,
                ];
            })
            ->filter()
            ->values();

        // Card 4: Ranking umum (top 5 dari semua kelas, terpisah per semester)
        // Ranking Ganjil
        $rankingUmumGanjil = BookHead::with(['siswa.kelas'])
            ->whereHas('siswa', function ($query) use ($thnAjaranAwal, $thnAjaranAkhir) {
                $query->where('thn_ajaran_awal', $thnAjaranAwal)
                    ->where('thn_ajaran_akhir', $thnAjaranAkhir)
                    ->where('deleted', '0');
            })
            ->where('semester', 'Ganjil')
            ->where('thn_ajaran_awal', $thnAjaranAwal)
            ->where('thn_ajaran_akhir', $thnAjaranAkhir)
            ->where('deleted', '0')
            ->orderBy('jml_nilai_akhir', 'desc')
            ->limit(5)
            ->get()
            ->map(function ($raport, $index) {
                $kelas = $raport->siswa->kelas ?? null;
                $kelasDisplay = 'N/A';

                if ($kelas) {
                    $roman = trim($kelas->nama_kelas_romawi ?? '');
                    $sub = trim($kelas->nama_kelas_sub ?? '');
                    $number = trim((string) ($kelas->nama_kelas_nomor ?? ''));

                    $romanWithSub = $roman;
                    if ($sub !== '') {
                        $romanWithSub = trim($romanWithSub . ' ' . $sub);
                    }

                    $numberWithSub = $number;
                    if ($sub !== '') {
                        $numberWithSub .= $sub;
                    }

                    $kelasDisplay = $romanWithSub;
                    if ($romanWithSub !== '' && $numberWithSub !== '') {
                        $kelasDisplay .= ' (' . $numberWithSub . ')';
                    }
                }

                return [
                    'rank' => $index + 1,
                    'nama' => $raport->siswa->nama_siswa ?? 'N/A',
                    'nis' => $raport->siswa->nis ?? 'N/A',
                    'kelas_display' => $kelasDisplay,
                    'jml_nilai_akhir' => $raport->jml_nilai_akhir ?? 0,
                ];
            });

        // Ranking Genap
        $rankingUmumGenap = BookHead::with(['siswa.kelas'])
            ->whereHas('siswa', function ($query) use ($thnAjaranAwal, $thnAjaranAkhir) {
                $query->where('thn_ajaran_awal', $thnAjaranAwal)
                    ->where('thn_ajaran_akhir', $thnAjaranAkhir)
                    ->where('deleted', '0');
            })
            ->where('semester', 'Genap')
            ->where('thn_ajaran_awal', $thnAjaranAwal)
            ->where('thn_ajaran_akhir', $thnAjaranAkhir)
            ->where('deleted', '0')
            ->orderBy('jml_nilai_akhir', 'desc')
            ->limit(5)
            ->get()
            ->map(function ($raport, $index) {
                $kelas = $raport->siswa->kelas ?? null;
                $kelasDisplay = 'N/A';

                if ($kelas) {
                    $roman = trim($kelas->nama_kelas_romawi ?? '');
                    $sub = trim($kelas->nama_kelas_sub ?? '');
                    $number = trim((string) ($kelas->nama_kelas_nomor ?? ''));

                    $romanWithSub = $roman;
                    if ($sub !== '') {
                        $romanWithSub = trim($romanWithSub . ' ' . $sub);
                    }

                    $numberWithSub = $number;
                    if ($sub !== '') {
                        $numberWithSub .= $sub;
                    }

                    $kelasDisplay = $romanWithSub;
                    if ($romanWithSub !== '' && $numberWithSub !== '') {
                        $kelasDisplay .= ' (' . $numberWithSub . ')';
                    }
                }

                return [
                    'rank' => $index + 1,
                    'nama' => $raport->siswa->nama_siswa ?? 'N/A',
                    'nis' => $raport->siswa->nis ?? 'N/A',
                    'kelas_display' => $kelasDisplay,
                    'jml_nilai_akhir' => $raport->jml_nilai_akhir ?? 0,
                ];
            });

        return view('pages.dashboard.index', compact('siswaPerKelas', 'selectedTahunAjaran', 'waliKelasProgress', 'rankingPerKelas', 'rankingUmumGanjil', 'rankingUmumGenap'));
    }
}
