<?php

namespace App\Exports;

use App\Models\Siswa;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\WithColumnWidths;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\Fill;

class SiswaExport implements FromCollection, WithHeadings, WithStyles, WithColumnWidths
{
    public function collection()
    {
        $siswas = Siswa::with(['kelas', 'pekerjaanAyah', 'pekerjaanIbu'])
            ->where('deleted', '0')
            ->orderBy('nama_siswa', 'asc')
            ->get();

        return $siswas->map(function ($siswa, $index) {
            return [
                $index + 1, // No
                $siswa->kelas->nama_kelas_nomor . ($siswa->kelas->nama_kelas_sub ?? ''), // Kelas
                $siswa->thn_ajaran_awal . '/' . $siswa->thn_ajaran_akhir, // Tahun Ajaran
                $siswa->nis, // NIS
                $siswa->nisn, // NISN
                $siswa->no_kip ?? '-', // No KIP
                $siswa->no_kk ?? '-', // No KK
                $siswa->nama_siswa, // Nama Siswa
                $siswa->jenis_kelamin, // Jenis Kelamin
                $siswa->tempat_lahir ?? '-', // Tempat Lahir
                $siswa->tanggal_lahir ? \Carbon\Carbon::parse($siswa->tanggal_lahir)->format('d-m-Y') : '-', // Tanggal Lahir
                $siswa->agama, // Agama
                $siswa->alamat ?? '-', // Alamat
                $siswa->no_hp_siswa ?? '-', // No HP Siswa
                $siswa->nama_ayah ?? '-', // Nama Ayah
                $siswa->pekerjaanAyah->nama_pekerjaan ?? '-', // Pekerjaan Ayah
                $siswa->nama_ibu ?? '-', // Nama Ibu
                $siswa->pekerjaanIbu->nama_pekerjaan ?? '-', // Pekerjaan Ibu
                $siswa->no_hp_ortu ?? '-', // No HP Ortu
                $siswa->angkatan, // Angkatan
            ];
        });
    }

    public function headings(): array
    {
        return [
            'No',
            'Kelas',
            'Tahun Ajaran',
            'NIS',
            'NISN',
            'No KIP',
            'No KK',
            'Nama Siswa',
            'Jenis Kelamin',
            'Tempat Lahir',
            'Tanggal Lahir',
            'Agama',
            'Alamat',
            'No HP Siswa',
            'Nama Ayah',
            'Pekerjaan Ayah',
            'Nama Ibu',
            'Pekerjaan Ibu',
            'No HP Ortu',
            'Angkatan',
        ];
    }

    public function styles(Worksheet $sheet)
    {
        // Style header
        $sheet->getStyle('A1:T1')->applyFromArray([
            'font' => [
                'bold' => true,
                'color' => ['rgb' => 'FFFFFF'],
            ],
            'fill' => [
                'fillType' => Fill::FILL_SOLID,
                'startColor' => ['rgb' => '4472C4'],
            ],
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_CENTER,
                'vertical' => Alignment::VERTICAL_CENTER,
            ],
            'borders' => [
                'allBorders' => [
                    'borderStyle' => Border::BORDER_THIN,
                ],
            ],
        ]);

        // Style all data cells
        $lastRow = $sheet->getHighestRow();
        $sheet->getStyle('A2:T' . $lastRow)->applyFromArray([
            'borders' => [
                'allBorders' => [
                    'borderStyle' => Border::BORDER_THIN,
                ],
            ],
            'alignment' => [
                'vertical' => Alignment::VERTICAL_CENTER,
            ],
        ]);

        // Center align specific columns
        $sheet->getStyle('A2:A' . $lastRow)->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER); // No
        $sheet->getStyle('B2:B' . $lastRow)->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER); // Kelas
        $sheet->getStyle('C2:C' . $lastRow)->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER); // Tahun Ajaran
        $sheet->getStyle('I2:I' . $lastRow)->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER); // Jenis Kelamin
        $sheet->getStyle('K2:K' . $lastRow)->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER); // Tanggal Lahir
        $sheet->getStyle('L2:L' . $lastRow)->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER); // Agama

        return [];
    }

    public function columnWidths(): array
    {
        return [
            'A' => 5,   // No
            'B' => 8,   // Kelas
            'C' => 15,  // Tahun Ajaran
            'D' => 15,  // NIS
            'E' => 15,  // NISN
            'F' => 15,  // No KIP
            'G' => 20,  // No KK
            'H' => 30,  // Nama Siswa
            'I' => 15,  // Jenis Kelamin
            'J' => 20,  // Tempat Lahir
            'K' => 15,  // Tanggal Lahir
            'L' => 12,  // Agama
            'M' => 40,  // Alamat
            'N' => 15,  // No HP Siswa
            'O' => 25,  // Nama Ayah
            'P' => 20,  // Pekerjaan Ayah
            'Q' => 25,  // Nama Ibu
            'R' => 20,  // Pekerjaan Ibu
            'S' => 15,  // No HP Ortu
            'T' => 10,  // Angkatan
        ];
    }
}
