<?php

namespace App\Exports;

use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\WithColumnWidths;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Events\AfterSheet;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\Fill;

class BukuIndukExport implements FromCollection, WithHeadings, WithStyles, WithColumnWidths, WithEvents
{
    protected $students;
    protected $mapelHeads;
    protected $bookHeads;
    protected $kelasName;
    protected $semester;
    protected $tahunAjaran;

    public function __construct($students, $mapelHeads, $bookHeads, $kelasName, $semester, $tahunAjaran)
    {
        $this->students = $students;
        $this->mapelHeads = $mapelHeads;
        $this->bookHeads = $bookHeads;
        $this->kelasName = $kelasName;
        $this->semester = $semester;
        $this->tahunAjaran = $tahunAjaran;
    }

    public function collection()
    {
        $data = [];

        foreach ($this->students as $index => $siswa) {
            $row = [];
            $row[] = $index + 1; // No
            $row[] = $siswa->nisn; // NISN
            $row[] = $siswa->nama_siswa; // Nama Siswa

            // Get book head for this student
            $bookHead = $this->bookHeads->get($siswa->id);
            $bookDetails = $bookHead ? $bookHead->details->keyBy(function ($detail) {
                return $detail->mapel_detail_id ?? 'head_' . $detail->mapel_head_id;
            }) : collect();

            // Add mapel values
            foreach ($this->mapelHeads as $mapelHead) {
                if ($mapelHead->details->isNotEmpty()) {
                    // Has details
                    foreach ($mapelHead->details as $detail) {
                        $detailData = $bookDetails->get($detail->id);
                        $row[] = $detailData ? $detailData->nilai_akhir : '';
                        $row[] = $detailData ? $detailData->deskripsi : '';
                    }
                } else {
                    // No details
                    $headData = $bookDetails->get('head_' . $mapelHead->id);
                    $row[] = $headData ? $headData->nilai_akhir : '';
                    $row[] = $headData ? $headData->deskripsi : '';
                }
            }

            $data[] = $row;
        }

        return collect($data);
    }

    public function headings(): array
    {
        // Row 1: Mapel Head names with colspan
        $row1 = ['No', 'NISN', 'Nama Siswa'];
        foreach ($this->mapelHeads as $mapelHead) {
            $row1[] = $mapelHead->nama_mapel;
            if ($mapelHead->details->isNotEmpty()) {
                // Add empty cells for colspan effect (will be merged later)
                for ($i = 1; $i < $mapelHead->details->count() * 2; $i++) {
                    $row1[] = '';
                }
            } else {
                // Add one empty cell for colspan of 2 (Nilai + Deskripsi)
                $row1[] = '';
            }
        }

        // Row 2: Mapel Detail names with colspan
        $row2 = ['', '', '']; // Empty for No, NISN, Nama Siswa (will use rowspan)
        foreach ($this->mapelHeads as $mapelHead) {
            if ($mapelHead->details->isNotEmpty()) {
                foreach ($mapelHead->details as $detail) {
                    $row2[] = $detail->nama_mapel_detail;
                    $row2[] = ''; // For colspan of 2 (Nilai + Deskripsi)
                }
            } else {
                // No details, use empty cells (will be merged with row 1)
                $row2[] = '';
                $row2[] = '';
            }
        }

        // Row 3: Nilai & Deskripsi labels
        $row3 = ['', '', '']; // Empty for No, NISN, Nama Siswa (will use rowspan)
        foreach ($this->mapelHeads as $mapelHead) {
            if ($mapelHead->details->isNotEmpty()) {
                foreach ($mapelHead->details as $detail) {
                    $row3[] = 'Nilai';
                    $row3[] = 'Deskripsi';
                }
            } else {
                $row3[] = 'Nilai';
                $row3[] = 'Deskripsi';
            }
        }

        return [$row1, $row2, $row3];
    }

    public function styles(Worksheet $sheet)
    {
        // Style for header rows (3 rows now)
        $sheet->getStyle('A1:' . $sheet->getHighestColumn() . '3')->applyFromArray([
            'font' => [
                'bold' => true,
            ],
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_CENTER,
                'vertical' => Alignment::VERTICAL_CENTER,
            ],
            'fill' => [
                'fillType' => Fill::FILL_SOLID,
                'startColor' => ['rgb' => 'E2E8F0'],
            ],
            'borders' => [
                'allBorders' => [
                    'borderStyle' => Border::BORDER_THIN,
                ],
            ],
        ]);

        // Style for data rows
        $lastRow = $this->students->count() + 3; // +3 because we have 3 header rows
        $sheet->getStyle('A4:' . $sheet->getHighestColumn() . $lastRow)->applyFromArray([
            'borders' => [
                'allBorders' => [
                    'borderStyle' => Border::BORDER_THIN,
                ],
            ],
        ]);

        // Center align No and NISN columns
        $sheet->getStyle('A4:B' . $lastRow)->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);

        return [];
    }

    public function columnWidths(): array
    {
        $widths = [
            'A' => 5,  // No
            'B' => 15, // NISN
            'C' => 30, // Nama Siswa
        ];

        // Add widths for mapel columns
        $column = 'D';
        foreach ($this->mapelHeads as $mapelHead) {
            if ($mapelHead->details->isNotEmpty()) {
                foreach ($mapelHead->details as $detail) {
                    $widths[$column++] = 12; // Nilai
                    $widths[$column++] = 40; // Deskripsi
                }
            } else {
                $widths[$column++] = 12; // Nilai
                $widths[$column++] = 40; // Deskripsi
            }
        }

        return $widths;
    }

    public function registerEvents(): array
    {
        return [
            AfterSheet::class => function(AfterSheet $event) {
                $sheet = $event->sheet->getDelegate();
                
                // Merge cells for No, NISN, Nama Siswa (rowspan 3)
                $sheet->mergeCells('A1:A3');
                $sheet->mergeCells('B1:B3');
                $sheet->mergeCells('C1:C3');

                // Merge cells for Mapel Heads and Details
                $currentCol = 'D'; // Start after Nama Siswa column
                
                foreach ($this->mapelHeads as $mapelHead) {
                    if ($mapelHead->details->isNotEmpty()) {
                        // Calculate colspan for mapel head
                        $colspan = $mapelHead->details->count() * 2;
                        $endCol = $this->getColumnLetter($this->getColumnIndex($currentCol) + $colspan - 1);
                        
                        // Merge mapel head across all its details (row 1)
                        $sheet->mergeCells($currentCol . '1:' . $endCol . '1');
                        
                        // Merge each detail across Nilai and Deskripsi (row 2)
                        $detailCol = $currentCol;
                        foreach ($mapelHead->details as $detail) {
                            $detailEndCol = $this->getColumnLetter($this->getColumnIndex($detailCol) + 1);
                            $sheet->mergeCells($detailCol . '2:' . $detailEndCol . '2');
                            $detailCol = $this->getColumnLetter($this->getColumnIndex($detailEndCol) + 1);
                        }
                        
                        $currentCol = $this->getColumnLetter($this->getColumnIndex($endCol) + 1);
                    } else {
                        // No details, merge mapel head with rowspan 2 and colspan 2
                        $endCol = $this->getColumnLetter($this->getColumnIndex($currentCol) + 1);
                        $sheet->mergeCells($currentCol . '1:' . $endCol . '2');
                        $currentCol = $this->getColumnLetter($this->getColumnIndex($endCol) + 1);
                    }
                }
            },
        ];
    }

    /**
     * Get column letter from index (0-based)
     */
    private function getColumnLetter($index)
    {
        $letter = '';
        while ($index >= 0) {
            $letter = chr($index % 26 + 65) . $letter;
            $index = floor($index / 26) - 1;
        }
        return $letter;
    }

    /**
     * Get column index from letter (0-based)
     */
    private function getColumnIndex($column)
    {
        $column = strtoupper($column);
        $index = 0;
        $length = strlen($column);
        for ($i = 0; $i < $length; $i++) {
            $index = $index * 26 + ord($column[$i]) - 64;
        }
        return $index - 1;
    }
}
