# Dokumentasi BookHeadDetailSeeder

## Deskripsi

`BookHeadDetailSeeder` adalah seeder yang digunakan untuk mengisi tabel `book_heads` dan `book_details` dengan data nilai siswa untuk semester Ganjil dan Genap. Seeder ini akan membuat record yang memenuhi syarat status **"Sudah Input"** pada halaman Buku Induk.

## Fitur

✅ Mengisi data nilai untuk **semua siswa aktif**  
✅ Membuat **BookHead** untuk semester **Ganjil** dan **Genap**  
✅ Membuat **BookDetail** untuk setiap mata pelajaran yang di-assign ke kelas siswa  
✅ Generate **nilai acak** (70-100)  
✅ Generate **deskripsi otomatis** berdasarkan nilai  
✅ Update **jml_nilai_akhir** di BookHead  
✅ Memastikan status **"Sudah Input"** tercapai  
✅ Skip jika data sudah ada (idempotent)  
✅ Transaction rollback jika ada error  

## Persyaratan

Sebelum menjalankan seeder ini, pastikan data berikut sudah ada di database:

1. ✅ **Siswa** - Minimal 1 siswa dengan status "Aktif"
2. ✅ **Kelas** - Kelas yang terhubung dengan siswa
3. ✅ **Tahun Ajaran** - Siswa harus memiliki tahun ajaran (thn_ajaran_awal & thn_ajaran_akhir)
4. ✅ **Mata Pelajaran** - MapelHead dan MapelDetail yang aktif
5. ✅ **MapelKelas** - Assignment mata pelajaran ke kelas

## Cara Menjalankan Seeder

### 1. Jalankan Seeder Secara Individual

```bash
php artisan db:seed --class=BookHeadDetailSeeder
```

### 2. Jalankan Melalui DatabaseSeeder

**Uncomment** baris seeder di `database/seeders/DatabaseSeeder.php`:

```php
public function run(): void
{
    $this->call([
        TahunAjaranSeeder::class,
        KelasSeeder::class,
        PekerjaanSeeder::class,
        MapelSeeder::class,
        // SiswaSeeder::class,
        UserSeeder::class,
        BookHeadDetailSeeder::class, // ← Uncomment this line
    ]);
}
```

Kemudian jalankan:

```bash
php artisan db:seed
```

### 3. Refresh Database dan Seed Ulang

**⚠️ WARNING: Ini akan menghapus semua data!**

```bash
php artisan migrate:fresh --seed
```

## Logika Seeder

### 1. Ambil Data Siswa Aktif

```php
$students = Siswa::where('deleted', '0')
    ->where('status', 'Aktif')
    ->with('kelas')
    ->get();
```

### 2. Untuk Setiap Siswa

- Ambil tahun ajaran siswa (`thn_ajaran_awal` dan `thn_ajaran_akhir`)
- Ambil mata pelajaran yang di-assign ke kelas siswa dari tabel `mapel_kelas`
- Filter mapel yang aktif (`status = 'Aktif'` dan `deleted = '0'`)

### 3. Buat BookHead

Untuk **semester Ganjil** dan **Genap**:

```php
BookHead::create([
    'siswa_id' => $siswa->id,
    'thn_ajaran_awal' => $thnAjaranAwal,
    'thn_ajaran_akhir' => $thnAjaranAkhir,
    'semester' => 'Ganjil', // atau 'Genap'
    'jml_nilai_akhir' => 0,
    'deleted' => '0',
]);
```

### 4. Buat BookDetail

Untuk setiap mata pelajaran:

**A. Jika MapelHead memiliki Details:**
```php
foreach ($mapelHead->details as $detail) {
    BookDetail::create([
        'book_head_id' => $bookHead->id,
        'mapel_head_id' => null,
        'mapel_detail_id' => $detail->id,
        'nilai_akhir' => rand(70, 100),
        'deskripsi' => 'Deskripsi...',
        'deleted' => '0',
    ]);
}
```

**B. Jika MapelHead tidak memiliki Details:**
```php
BookDetail::create([
    'book_head_id' => $bookHead->id,
    'mapel_head_id' => $mapelHead->id,
    'mapel_detail_id' => null,
    'nilai_akhir' => rand(70, 100),
    'deskripsi' => 'Deskripsi...',
    'deleted' => '0',
]);
```

### 5. Update Total Nilai

```php
$bookHead->update([
    'jml_nilai_akhir' => $totalNilai
]);
```

## Generate Nilai dan Deskripsi

### Nilai (Random 70-100)

```php
private function generateNilai(): int
{
    return rand(70, 100);
}
```

### Deskripsi (Berdasarkan Nilai)

**Nilai 90-100 (Sangat Baik):**
- "Siswa menunjukkan pemahaman yang sangat baik dalam {mapel}..."
- "Prestasi yang sangat memuaskan dalam {mapel}..."

**Nilai 80-89 (Baik):**
- "Siswa menunjukkan pemahaman yang baik dalam {mapel}..."
- "Prestasi yang baik dalam {mapel}..."

**Nilai 70-79 (Cukup):**
- "Siswa menunjukkan pemahaman yang cukup dalam {mapel}..."
- "Prestasi yang cukup dalam {mapel}..."

## Syarat Status "Sudah Input"

Berdasarkan logika di `buku_induk/index.blade.php`, status "Sudah Input" tercapai jika:

✅ Ada **BookHead** untuk siswa  
✅ Untuk setiap mata pelajaran yang di-assign:
   - Jika mapel memiliki **details**: Setiap detail harus memiliki `BookDetail` dengan `nilai_akhir` dan `deskripsi` **tidak kosong**
   - Jika mapel **tidak memiliki details**: MapelHead harus memiliki `BookDetail` dengan `nilai_akhir` dan `deskripsi` **tidak kosong**

## Output Seeder

Setelah seeder berhasil dijalankan, Anda akan melihat output seperti ini:

```
✅ Seeder completed successfully!
📊 Summary:
   - Total BookHeads created: 60
   - Total BookDetails created: 420
```

**Keterangan:**
- 30 siswa × 2 semester = 60 BookHeads
- 30 siswa × 2 semester × 7 mapel = 420 BookDetails (contoh)

## Log

Seeder ini akan mencatat aktivitas di log Laravel:

```
storage/logs/laravel.log
```

Contoh log:
```
[2025-10-28 10:00:00] local.INFO: Starting BookHeadDetailSeeder...
[2025-10-28 10:00:00] local.INFO: Found 30 active students
[2025-10-28 10:00:01] local.INFO: Processing student: Ahmad Rizki (NISN: 0012345001)
[2025-10-28 10:00:01] local.INFO: Found 7 mapel heads for student Ahmad Rizki
[2025-10-28 10:00:01] local.INFO: Created BookHead for Ahmad Rizki - Semester ganjil
[2025-10-28 10:00:02] local.INFO: Created 7 BookDetails for Ahmad Rizki - Semester ganjil with total nilai: 630
```

## Troubleshooting

### Error: No active students found

**Penyebab:** Tidak ada siswa dengan status "Aktif" di database

**Solusi:**
1. Seed data siswa terlebih dahulu
2. Atau import siswa dari Excel
3. Pastikan kolom `status` = 'Aktif' dan `deleted` = '0'

```bash
php artisan db:seed --class=SiswaSeeder
```

### Error: No mapel assigned to class

**Penyebab:** Tidak ada mata pelajaran yang di-assign ke kelas siswa di tabel `mapel_kelas`

**Solusi:**
1. Assign mata pelajaran ke kelas melalui menu **Mata Pelajaran** di aplikasi
2. Atau seed data MapelKelas

### Error: BookHead already exists

**Penyebab:** Data BookHead untuk siswa dan semester tertentu sudah ada

**Solusi:**
- Seeder akan **skip** otomatis (idempotent)
- Jika ingin re-seed, hapus data BookHead dan BookDetail terlebih dahulu:

```sql
DELETE FROM book_details;
DELETE FROM book_heads;
```

Atau via Artisan:

```bash
php artisan tinker
>>> App\Models\BookHead::truncate();
>>> App\Models\BookDetail::truncate();
```

### Error: Transaction rollback

**Penyebab:** Ada error saat proses seeding

**Solusi:**
1. Cek log di `storage/logs/laravel.log`
2. Pastikan foreign key constraints terpenuhi
3. Pastikan data master (siswa, kelas, mapel) sudah ada

## Verifikasi Hasil

### 1. Melalui Aplikasi

Buka halaman **Buku Induk** di aplikasi:
```
http://localhost:8000/buku-induk
```

Pastikan:
- ✅ Status Input menampilkan **"Sudah Input"** (hijau)
- ✅ Nilai muncul di setiap kolom mapel
- ✅ Deskripsi muncul di setiap kolom mapel

### 2. Melalui Database

```sql
-- Cek jumlah BookHead
SELECT COUNT(*) as total_book_heads FROM book_heads WHERE deleted = '0';

-- Cek jumlah BookDetail
SELECT COUNT(*) as total_book_details FROM book_details WHERE deleted = '0';

-- Cek status input per siswa
SELECT 
    s.nama_siswa,
    bh.semester,
    COUNT(bd.id) as jumlah_detail,
    SUM(bd.nilai_akhir) as total_nilai
FROM book_heads bh
JOIN siswas s ON bh.siswa_id = s.id
LEFT JOIN book_details bd ON bh.id = bd.book_head_id
WHERE bh.deleted = '0'
GROUP BY bh.id, s.nama_siswa, bh.semester
ORDER BY s.nama_siswa, bh.semester;
```

### 3. Melalui Artisan Tinker

```bash
php artisan tinker
```

```php
// Cek total BookHead
>>> App\Models\BookHead::where('deleted', '0')->count();

// Cek total BookDetail
>>> App\Models\BookDetail::where('deleted', '0')->count();

// Cek detail BookHead dengan relasi
>>> App\Models\BookHead::with('details')->first();
```

## Best Practices

1. **Backup Database** sebelum menjalankan seeder di production
2. **Test di development** terlebih dahulu
3. **Gunakan transaction** (sudah diimplementasi di seeder)
4. **Check log** setelah seeding untuk memastikan tidak ada error
5. **Verifikasi hasil** melalui aplikasi dan database

## Technical Details

### Models yang Digunakan

- `Siswa` - Data siswa
- `BookHead` - Header nilai siswa per semester
- `BookDetail` - Detail nilai per mata pelajaran
- `MapelHead` - Mata pelajaran utama
- `MapelDetail` - Detail mata pelajaran
- `MapelKelas` - Assignment mapel ke kelas

### Database Tables

- `siswas` - Tabel siswa
- `book_heads` - Tabel header nilai
- `book_details` - Tabel detail nilai
- `mapel_heads` - Tabel mapel utama
- `mapel_details` - Tabel detail mapel
- `mapel_kelas` - Tabel assignment mapel ke kelas

### Foreign Keys

```
book_heads.siswa_id → siswas.id
book_details.book_head_id → book_heads.id
book_details.mapel_head_id → mapel_heads.id (nullable)
book_details.mapel_detail_id → mapel_details.id (nullable)
```

## Contoh Use Case

### Use Case 1: Fresh Installation

```bash
# 1. Migrate database
php artisan migrate

# 2. Seed master data
php artisan db:seed --class=TahunAjaranSeeder
php artisan db:seed --class=KelasSeeder
php artisan db:seed --class=PekerjaanSeeder
php artisan db:seed --class=MapelSeeder

# 3. Import or seed siswa
php artisan db:seed --class=SiswaSeeder
# atau via import Excel di aplikasi

# 4. Seed BookHead dan BookDetail
php artisan db:seed --class=BookHeadDetailSeeder
```

### Use Case 2: Development Testing

```bash
# Fresh install dengan seed lengkap
php artisan migrate:fresh
php artisan db:seed
php artisan db:seed --class=SiswaSeeder
php artisan db:seed --class=BookHeadDetailSeeder
```

### Use Case 3: Update Existing Data

```bash
# Hanya seed BookHead dan BookDetail untuk data baru
php artisan db:seed --class=BookHeadDetailSeeder
```

## Author

Created for E-Raport-Book Project  
Date: 2025-10-28
